/* Copyright (c) 2004 Dirk Jagdmann <doj@cubic.org>

This software is provided 'as-is', without any express or implied
warranty. In no event will the authors be held liable for any damages
arising from the use of this software.

Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute it
freely, subject to the following restrictions:

    1. The origin of this software must not be misrepresented; you
       must not claim that you wrote the original software. If you use
       this software in a product, an acknowledgment in the product
       documentation would be appreciated but is not required.

    2. Altered source versions must be plainly marked as such, and
       must not be misrepresented as being the original software.

    3. This notice may not be removed or altered from any source
       distribution. */

// $Header: /code/doj/bezier.hpp,v 1.6 2005/02/26 19:31:33 doj Exp $

#ifndef BEZIER__HPP
#define BEZIER__HPP

#include <cmath>
#include <vector>

namespace doj
{

  /**
     Bezier makes an interpolation between objects.<p>

     You can interpolate objects of template type T. They are stored
     in a container of template type Container, which defaults to the
     STL-deque. Some internal calculations are done with floating
     point type FT which defaults to C-float. If you want to calculate
     with double precision make this template parameter a double.<p>

     Before interpolating you have to fill the container. Use
     container() to get a reference to the container or push_back to
     fill it. Then you initialize the interpolation by calling
     init(). You have to call init() every time you make changes to
     the container a and want to interpolate. (In case you forgot
     this, the interpolation function will call init() for you, but
     this involves a heavy calculation penalty for the first call)<p>

     Interpolation is undefined for an empty container.<br>
     Interpolation is constant for 1 element (the element itself).<br>
     Interpolation is linear for 2 elements.<br>
     Interpolation if Bezier-spline for 3 and more elemts.<br>

     The template type T has to provide the following methods:<br>
     T::T() - default constructor<br>
     T& operator=(const T&) - assignment<br>

     T operator + (const T&, const T&) - addition of two Ts.<br>
     T operator - (const T&, const T&) - subtraktion of two Ts.<br>
     T operator * (const T&, const FT&) - a multiplication with a float-type.<br>
  */
  template <typename T , typename FT = double, typename Cont = std::vector<T> >
  class Bezier
  {
  public:
    /// Container type
    typedef Cont Container;
    /// iterator for Container
    typedef typename Container::iterator iterator;
    /// const_interator for Container
    typedef typename Container::const_iterator const_iterator;

  private:
    /// user accessible container used to store the points to interpolate
    Container a;
    /// size of a after init()
    size_t asize;
    /// private vector storing the bezier coefficients
    std::vector<T> P;
    /// tells if P was initialized.
    bool initialized;

  public:
    Bezier () :
      initialized(false)
    {
    }

    /// initialize the current element set for interpolation. @return this
    void init()
    {
      initialized=true;
      asize=a.size();

      if(asize < 3)
	return;

      const int N = asize*3-2;
      P.resize(N);

      P[0] = a[0];
      P[1] = a[0];
      P[N-2] = a[asize-1];
      P[N-1] = a[asize-1];

      size_t i;
      for(i=1; i < asize-1; i++)
	P[i*3] = a[i];

      for(i=1; i < asize-1; i++)
	{
	  const int ii = 3*i;
	  P[ii-1] = (P[ii-2] - P[ii+3] + P[ii]*static_cast<FT>(2.0)) * static_cast<FT>(0.5);
	  P[ii+1] = P[ii]*static_cast<FT>(2.0) - P[ii-1];
	}
    }

    /**
       make a interpolation. A new T is returned with the interpolated value for the given t.
       @param t time offset into the element container. 0 <= t <= 1
       @return the interpolated T
    */
    T operator()(FT t)
    {
      if(asize==0)
	return T();

      if(!initialized)
	init();

      if(t<0.0) t=0.0;
      else if (t>1.0) t=1.0;

      if(t==0.0)
	return a[0];
      if(t==1.0)
	return a[asize-1];

      if(asize==1)
	return a[0];

      if(asize==2)
	return a[0]*static_cast<FT>(1.0-t) + a[1]*t;

      const int segments = asize-1;
      const int x = static_cast<int>(t*segments) * 3;
      const FT t2 = ::fmod(t, 1.0/segments) * segments;
      const FT tt = t2*t2;
      const FT mt = 1.0-t2;
      const FT mtmt = mt*mt;
      return P[x]*(mt*mtmt) + P[x+1]*(t2*mtmt*3) + P[x+2]*(tt*mt*3) + P[x+3]*(t2*tt);
    }

    /// @return the container used to store the elements
    Container& container()
    {
      initialized=false;
      return a;
    }

    /// @return the container used to store the elements
    const Container& container() const
    {
      return a;
    }

    /// @return element i of Container
    T& operator[](const int i)
    {
      initialized=false;
      return a[i];
    }

    /// @return element i of Container
    const T& operator[](const int i) const
    {
      return a[i];
    }

    /// @return size of Container
    size_t size() const
    {
      return a.size();
    }

    /// add t to back of Container
    void push_back(const T& t)
    {
      initialized=false;
      a.push_back(t);
    }

    /// add t to front of Container
    void push_front(const T& t)
    {
      initialized=false;
      a.push_front(t);
    }

    /// @return the first element of Container
    T& front()
    {
      initialized=false;
      return a.front();
    }

    /// @return the last element of Container
    T& back()
    {
      initialized=false;
      return a.back();
    }

    /// remove the first element of Container
    void pop_front()
    {
      initialized=false;
      a.pop_front();
    }

    /// remove the last element of Container
    void pop_back()
    {
      initialized=false;
      a.pop_back();
    }

    /// clear Container
    void clear()
    {
      initialized=false;
      a.clear();
    }

    /// @return a.begin()
    iterator begin()
    {
      initialized=false;
      return a.begin();
    }

    /// @return a.begin()
    const_iterator begin() const
    {
      return a.begin();
    }

    /// @return a.end()
    iterator end()
    {
      initialized=false;
      return a.end();
    }

    /// @return a.end()
    const_iterator end() const
    {
      return a.end();
    }

  };

}

#endif
