/* Copyright (c) 2004 Dirk Jagdmann <doj@cubic.org>

This software is provided 'as-is', without any express or implied
warranty. In no event will the authors be held liable for any damages
arising from the use of this software.

Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute it
freely, subject to the following restrictions:

    1. The origin of this software must not be misrepresented; you
       must not claim that you wrote the original software. If you use
       this software in a product, an acknowledgment in the product
       documentation would be appreciated but is not required.

    2. Altered source versions must be plainly marked as such, and
       must not be misrepresented as being the original software.

    3. This notice may not be removed or altered from any source
       distribution. */

// $Header: /code/doj/color.cpp,v 1.7 2006/07/18 18:59:08 doj Exp $

#include <algorithm>
#include <iostream>
#include <string>

using namespace std;

#include <color.hpp>
#include <constant.hpp>

namespace doj
{

  //// RGBA ///////////////////////////////////////////////////////////

  rgba& rgba::operator=(const cmy& cm)
  {
    r=1.0-cm.c;
    g=1.0-cm.m;
    b=1.0-cm.y;
    a=1;
    return *this;
  }

  rgba& rgba::operator=(const cmyk& cm)
  {
    r=1.0-std::min(1.0,cm.c*(1.0-cm.k)+cm.k);
    g=1.0-std::min(1.0,cm.m*(1.0-cm.k)+cm.k);
    b=1.0-std::min(1.0,cm.y*(1.0-cm.k)+cm.k);
    a=1;
    return *this;
  }

  rgba& rgba::operator=(const hsv& hs)
  {
    a=1;
    if(hs.s==0.0)
      {
	r=g=b=0;
	return *this;
      }

    colorfloattype hh=hs.h;
    while(hh>360.0)
      hh-=360.0;
    while(hh<0.0)
      hh+=360.0;
    hh/=60.0;

    const int i=static_cast<int>(floor(hh));
    const colorfloattype f=hh-i;
    const colorfloattype p=hs.v*(1.0-hs.s);
    const colorfloattype q=hs.v*(1.0-hs.s*f);
    const colorfloattype t=hs.v*(1.0-hs.s*(1.0-f));

    switch(i) {
    case 0:
      r=hs.v; g=t; b=p; break;
    case 1:
      r=q; g=hs.v; b=p; break;
    case 2:
      r=p; g=hs.v; b=t; break;
    case 3:
      r=p; g=q; b=hs.v; break;
    case 4:
      r=t; g=p; b=hs.v; break;
    case 5:
    default:
      r=hs.v; g=p; b=q; break;
    }

    a=1;
    return *this;
  }

  static colorfloattype HueToRGB(const colorfloattype n1, const colorfloattype n2, colorfloattype hue)
  {
    // range check: note values passed add/subtract thirds of range
    while (hue < 0.0)
      hue += 1.0;

    while (hue > 1.0)
      hue -= 1.0;

    // return r,g, or b value from this tridrant
    if (hue < 1.0/6.0)
      return ( n1 + (((n2-n1)*hue+1.0/12.0)/(1.0/6.0)) );
    if (hue < 1.0/2.0)
      return ( n2 );
    if (hue < 2.0/3.0)
      return ( n1 + (((n2-n1)*((2.0/3.0)-hue)+1.0/12.0)/(1.0/6.0)) );
    else
      return ( n1 );
  }

  rgba& rgba::operator=(const hls& hl)
  {
    if (hl.s == 0.0)		// achromatic case
      r=g=b=hl.l;
    else			// chromatic case
      {
	// set up magic numbers
	colorfloattype Magic2;
	if (hl.l <= .5)
	  Magic2 = hl.l*hl.s + .5;
	else
	  Magic2 = hl.l + hl.s - hl.l*hl.s + .5;
	const colorfloattype Magic1 = 2.0*hl.l - Magic2;

	// get RGB, change units from HLSMAX to RGBMAX
	r = HueToRGB(Magic1,Magic2,hl.h+1.0/3.0) + .5;
	g = HueToRGB(Magic1,Magic2,hl.h) + .5;
	b = HueToRGB(Magic1,Magic2,hl.h-1.0/3.0) + .5;
      }

    a=1;
    return *this;

#if 0
Procedure HLStoRGBf (H,L,S: Single; Var rb,gb,bb: Byte);
{ H: 0..360, L: 0..1, S: 0..1; rb,gb,bb: 0..255; Foley + van Dam }
 Var min,max,mmm,hue,col : Single; r,g,b: Integer;
 Begin
 While H>360.0 Do H:=H-360.0;
 While H<0.0   Do H:=H+360.0;
 If S<0 Then S:=0 Else If S>1 Then S:=1;
 If L<0 Then L:=0 Else If L>1 Then L:=1;
 If L<=0.5 Then max:=L*(1.0+S) Else max:=L+S-L*S;
 min:=2*L-max; mmm:=(max-min)/60;
   hue:=H+120.0;
    If hue>360.0 Then hue:=hue-360.0;
         If hue< 60.0 Then col:=min+mmm*hue
    Else If hue<180.0 Then col:=max
    Else If hue<240.0 Then col:=min+mmm*(240.0-hue)
    Else                   col:=min;
   r:=Round(255.0*col); LimitP(r,255); rb:=r;
   hue:=H;
         If hue< 60.0 Then col:=min+mmm*hue
    Else If hue<180.0 Then col:=max
    Else If hue<240.0 Then col:=min+mmm*(240.0-hue)
    Else                   col:=min;
   g:=Round(255.0*col); LimitP(g,255); gb:=g;
   hue:=H-120.0;
    If hue<0.0 Then hue:=hue+360.0;
         If hue< 60.0 Then col:=min+mmm*hue
    Else If hue<180.0 Then col:=max
    Else If hue<240.0 Then col:=min+mmm*(240.0-hue)
    Else                   col:=min;
   b:=Round(255.0*col); LimitP(b,255); bb:=b;
End;
#endif
  }

  rgba& rgba::operator=(const hwb& HWB)
  {
    const colorfloattype v = 1.0 - HWB.b;
    const float H=HWB.h*6.0;
    const int i = static_cast<int>(::floor(H));
    colorfloattype f = H - i;
    if (i & 1) f = 1.0 - f; // if i is odd
    const colorfloattype n = HWB.w + f * (v - HWB.w); // linear interpolation between w and v
    switch (i) {
    default:
    case 6:
    case 0: r=v; g=n; b=HWB.w; break;
    case 1: r=n; g=v; b=HWB.w; break;
    case 2: r=HWB.w; g=v; b=n; break;
    case 3: r=HWB.w; g=n; b=v; break;
    case 4: r=n; g=HWB.w; b=v; break;
    case 5: r=v; g=HWB.w; b=n; break;
    }
    return *this;
  }

  rgba& rgba::operator=(const gray& gr)
  {
    r=g=b=gr.g;
    a=gr.a;
    return *this;
  }

  rgba& rgba::operator=(const xyz& xy)
  {
#if 0
 [ R709 ] [ 3.240479 -1.53715  -0.498535 ] [ X ]
 [ G709 ]=[-0.969256  1.875991  0.041556 ]*[ Y ]
 [ B709 ] [ 0.055648 -0.204043  1.057311 ] [ Z ]
#endif
#if 0
[R] = [  2.739 -1.145 -0.424 ] [X]
[G] = [ -1.119  2.029  0.033 ] [Y]
[B] = [  0.138 -0.333  1.105 ] [Z]
#endif
    return *this;
  }

  rgba& rgba::operator=(const yuv& yu)
  {
    r= yu.y +/*0.000*yu.u +*/ 1.140*yu.v;
    g= yu.y -  0.396*yu.u -   0.581*yu.v;
    b= yu.y +  2.029*yu.u /*+ 0.000*yu.v*/;
    a=1;
    return *this;
  }

  rgba& rgba::operator=(const yiq& yi)
  {
    r = yi.y + 0.956*yi.i + 0.621*yi.q;
    g = yi.y - 0.272*yi.i - 0.647*yi.q;
    b = yi.y - 1.105*yi.i + 1.702*yi.q;
    a=1;
    return *this;
  }

  rgba& rgba::operator=(const ycrcb& yc)
  {
    r = yc.y + 0.0000*yc.cb + 1.4022*yc.cr;
    g = yc.y - 0.3456*yc.cb - 0.7145*yc.cr;
    b = yc.y + 1.7710*yc.cb + 0.0000*yc.cr;
    a=1;
    return *this;
  }

  rgba& rgba::dreidreizwei(uint8_t i)
  {
    b=(i&0x3)/4.0; i>>=2;
    g=(i&0x7)/8.0; i>>=3;
    r=(i&0x7)/8.0;
    a=1;
    return *this;
  }

  rgba& rgba::fuenffuenffuenf(uint16_t i)
  {
    b=(i&0x1F)/32.0; i>>=5;
    g=(i&0x1F)/32.0; i>>=5;
    r=(i&0x1F)/32.0;
    a=1;
    return *this;
  }

  rgba& rgba::fuenfsechsfuenf(uint16_t i)
  {
    b=(i&0x1F)/32.0; i>>=5;
    g=(i&0x3F)/64.0; i>>=6;
    r=(i&0x1F)/32.0;
    a=1;
    return *this;
  }

  rgba& rgba::argb(uint32_t i)
  {
    b=(i&0xFF)/255.0; i>>=8;
    g=(i&0xFF)/255.0; i>>=8;
    r=(i&0xFF)/255.0; i>>=8;
    a=(i&0xFF)/255.0;
    return *this;
  }

  std::ostream& rgba::dump(std::ostream& os) const
  {
    return os << "<rgba> " << r << " " << g << " " << b << " " << a << " </rgba>";
  }

  std::istream& rgba::scan(std::istream& is)
  {
    string s;
    is >> s;
    if(s!="<rgba>")
      {
#ifdef DOJDEBUG
	cerr << "rgba::scan(): <rgba> tag expected, but got '" << s << "' aborting..." << endl;
#endif
	return is;
      }

    is >> r >> g >> b >> a;

    is >> s;
    if(s!="</rgba>")
      {
#ifdef DOJDEBUG
	cerr << "rgba::scan(): </rgba> tag expected." << endl;
#endif
      }

    return is;
  }

  rgba& rgba::invert()
  {
    r=1.0-r; g=1.0-g; b=1.0-b;
    return *this;
  }

  //// HSV /////////////////////////////////////////////////////

  hsv& hsv::operator=(const rgba& rgb)
  {
    const colorfloattype ma=std::max(rgb.r, std::max(rgb.g, rgb.b));
    v=ma;

    const colorfloattype mi=std::min(rgb.r, std::min(rgb.g, rgb.b));
    const colorfloattype delta=ma-mi;

    if(ma!=0)
      s=(delta)/ma;
    else
      {
	s=v=h=0;
	return *this;
      }

    if(rgb.r==ma)
      h=(rgb.g-rgb.b)/delta;
    else if(rgb.g==ma)
      h=2.0+(rgb.b-rgb.r)/delta;
    else
      h=4.0+(rgb.r-rgb.g)/delta;
    h*=60.0;
    while(h<0.0)
      h+=360.0;
    while(h>360.0)
      h-=360.0;

    return *this;
  }

  //// HWB //////////////////////////////////////////////////

  hwb& hwb::operator=(const rgba& rgb)
  {
    const colorfloattype V = std::max(rgb.r, std::min(rgb.g, rgb.b));
    w = std::min(rgb.r, std::min(rgb.g, rgb.b));
    b = 1.0 - V;
    if (V == w)
      {
	h=0;
	return *this;
      }
    const colorfloattype F = (rgb.r == w) ? rgb.g - rgb.b : ((rgb.g == w) ? rgb.b - rgb.r : rgb.r - rgb.g);
    const colorfloattype I = (rgb.r == w) ? 3 : ((rgb.g == w) ? 5 : 1);
    h=I - F /(V - w);
    h/=6.0;
    return *this;
  }

  //// CMY ///////////////////////////////////////////////////

  cmy& cmy::operator=(const rgba& rgb)
  {
    c=1.0-rgb.r;
    m=1.0-rgb.g;
    y=1.0-rgb.b;
    return *this;
  }

  cmy& cmy::operator=(const cmyk& cm)
  {
    c=std::min(1.0, cm.c*(1.0-cm.k)+cm.k);
    m=std::min(1.0, cm.m*(1.0-cm.k)+cm.k);
    y=std::min(1.0, cm.y*(1.0-cm.k)+cm.k);
    return *this;
  }

  /// CMYK ///////////////////////////////////////////

  cmyk& cmyk::operator=(const rgba& rgb)
  {
    k=std::min(1.0-rgb.r, std::min(1.0-rgb.g, 1.0-rgb.b));
    c=(1.0-rgb.r-k)/(1.0-k);
    m=(1.0-rgb.g-k)/(1.0-k);
    y=(1.0-rgb.b-k)/(1.0-k);
    return *this;
  }

  cmyk& cmyk::operator=(const cmy& cm)
  {
    k=std::min(cm.c, std::min(cm.m,cm.y));
    if(k==1.0)
      c=m=y=0;
    else
      {
	c=(cm.c-k)/(1.0-k);
	m=(cm.m-k)/(1.0-k);
	y=(cm.y-k)/(1.0-k);
      }
    return *this;
  }

  //// YCrCb //////////////////////////////

  ycrcb& ycrcb::operator=(const rgba& rgb)
  {
    y = 0.2989*rgb.r + 0.5866*rgb.g + 0.1145*rgb.b;
    cb=-0.1687*rgb.r - 0.3312*rgb.g + 0.5000*rgb.b;
    cr= 0.5000*rgb.r - 0.4183*rgb.g - 0.0816*rgb.b;
    return *this;
  }

  //// YUV ////////////////////////////////////////

  // for details see: http://www.fourcc.org/fccyvrgb.php (~doj/archive/fccyvrgb.html)
  yuv& yuv::operator=(const rgba& rgb)
  {
#if 1
    y =  0.299*rgb.r + 0.587*rgb.g + 0.114*rgb.b;
    u = -0.147*rgb.r - 0.289*rgb.g + 0.436*rgb.b;
    v =  0.615*rgb.r - 0.515*rgb.g - 0.100*rgb.b;
#else
    y =  0.299*rgb.r + 0.587*rgb.g + 0.114*rgb.b;
    u = (rgb.b-y)*0.565;
    v = (rgb.r-y)*0.713;
#endif
    return *this;
  }

  yuv& yuv::operator=(const ycrcb& ycc)
  {
    y=ycc.y;
    u=ycc.cb*0.492111;
    v=ycc.cr*0.877283;
    return *this;
  }

  yuv& yuv::operator=(const yiq& yi)
  {
    y = yi.y;
    u = -1.1270*yi.i + 1.8050*yi.q;
    v =  0.9489*yi.i + 0.6561*yi.q;
    return *this;
  }

  /// Gray //////////////////////////////////////////

  gray& gray::operator=(const rgba& rgb)
  {
    g=0.3*rgb.r+0.59*rgb.g+0.11*rgb.b;
    a=rgb.a;
    return *this;
  }

  //// HLS /////////////////////////////////

  hls& hls::operator=(const rgba& rgb)
  {
    // calculate lightness
    const colorfloattype cMax = std::max( std::max(rgb.r, rgb.g), rgb.b);
    const colorfloattype cMin = std::min( std::min(rgb.r, rgb.g), rgb.b);
    l = ( cMax+cMin + 1.0 )/2.0;

    if (cMax == cMin)		// r=g=b --> achromatic case
      s = h = 0;		// saturation
    else			// chromatic case
      {
	// saturation
	if (l <= .5)
	  s = (cMax-cMin + (cMax+cMin)/2.0 ) / (cMax+cMin);
	else
	  s = (cMax-cMin + (2.0-cMax-cMin)/2.0 ) / (2.0-cMax-cMin);

	// hue
	const colorfloattype Rdelta = ( (cMax-rgb.r)/6.0 + (cMax-cMin)/2.0 ) / (cMax-cMin);
	const colorfloattype Gdelta = ( (cMax-rgb.g)/6.0 + (cMax-cMin)/2.0 ) / (cMax-cMin);
	const colorfloattype Bdelta = ( (cMax-rgb.b)/6.0 + (cMax-cMin)/2.0 ) / (cMax-cMin);

	if (rgb.r == cMax)
	  h = Bdelta - Gdelta;
	else if (rgb.g == cMax)
	  h = 1.0/3.0 + Rdelta - Bdelta;
	else // rgb.b == cMax
	  h = 2.0/3.0 + Gdelta - Rdelta;

	while (h < 0.0)
	  h += 1.0;
	while (h > 1.0)
	  h -= 1.0;
      }

    return *this;
  }

  /// XYZ ////////////////////////////////////////

  xyz& xyz::operator=(const rgba& rgb)
  {
    x = 0.412453*rgb.r + 0.35758 *rgb.g + 0.180423*rgb.b;
    y = 0.212671*rgb.r + 0.71516 *rgb.g + 0.072169*rgb.b;
    z = 0.019334*rgb.r + 0.119193*rgb.g + 0.950227*rgb.b;
    return *this;
  }

  xyz& xyz::operator=(const yes& ye)
  {
    x=0.782*ye.y - 0.466*ye.e + 0.138*ye.s;
    y=1.000*ye.y + 0.000*ye.e + 0.000*ye.s;
    z=0.671*ye.y - 0.237*ye.e - 1.133*ye.s;
    return *this;
  }

  //// LCH ////////////////////////////////////////

  lch& lch::operator=(const lab& la)
  {
    l = la.l;
    c = sqrt(la.a*la.a+la.b*la.b);
    colorfloattype K=0;
    if(la.a>0.0 && la.b<0.0)
      K=1;
    else if(la.a<0.0 && la.b<0.0)
      K=2;
    else if(la.a<0.0 && la.b>0.0)
      K=3;

    if(la.a==0.0)
      h=0;
    else
      h=(atan(la.b/la.a) + K*M_PI_2) / (2.0*M_PI);
    if(h<0.0)
      h+=static_cast<colorfloattype>(M_PI_2);

    return *this;
  }

  //// YIQ //////////////////////////////////////

  yiq& yiq::operator=(const rgba& rgb)
  {
    y = 0.299*rgb.r + 0.587*rgb.g + 0.114*rgb.b;
    i = 0.596*rgb.r - 0.274*rgb.g - 0.322*rgb.b;
    q = 0.212*rgb.r - 0.523*rgb.g + 0.311*rgb.b;
    return *this;
  }

  yiq& yiq::operator=(const yuv& yu)
  {
    y = yu.y;
    i = -0.2676*yu.u + 0.7361*yu.v;
    q =  0.3869*yu.u + 0.4596*yu.v;
    return *this;
  }

  /// YES ////////////////////////////////////////////////////////

  yes& yes::operator=(const xyz& xy)
  {
    y= 0.000*xy.x + 1.000*xy.y + 0.000*xy.z;
    e=-2.019*xy.x + 1.743*xy.y - 0.246*xy.z;
    s= 0.423*xy.x + 0.227*xy.y - 0.831*xy.z;
    return *this;
  }

  // ripped from http://msdn.microsoft.com/library/default.asp?url=/library/en-us/wceddraw/html/_dxce_converting_between_yuv_and_rgb.asp
  void rgb2yuv(const uint8_t r, const uint8_t g, const uint8_t b, uint8_t& y, uint8_t& u, uint8_t v)
  {
    y = ( (  66 * r + 129 * g +  25 * b + 128) >> 8) +  16;
    u = ( ( -38 * r -  74 * g + 112 * b + 128) >> 8) + 128;
    v = ( ( 112 * r -  94 * g -  18 * b + 128) >> 8) + 128;
  }
  namespace { inline int clip(int i) { if(i<0) return 0; else if(i>255) return 255; return i; } }
  void yuv2rgb(const uint8_t y, const uint8_t u, const uint8_t v, uint8_t& r, uint8_t& g, uint8_t b)
  {
    const int c = y - 16;
    const int d = u - 128;
    const int e = v - 128;
    r = clip(( 298 * c           + 409 * e + 128) >> 8);
    g = clip(( 298 * c - 100 * d - 208 * e + 128) >> 8);
    b = clip(( 298 * c + 516 * d           + 128) >> 8);
  }
};
