/* Copyright (c) 2004 Dirk Jagdmann <doj@cubic.org>

This software is provided 'as-is', without any express or implied
warranty. In no event will the authors be held liable for any damages
arising from the use of this software.

Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute it
freely, subject to the following restrictions:

    1. The origin of this software must not be misrepresented; you
       must not claim that you wrote the original software. If you use
       this software in a product, an acknowledgment in the product
       documentation would be appreciated but is not required.

    2. Altered source versions must be plainly marked as such, and
       must not be misrepresented as being the original software.

    3. This notice may not be removed or altered from any source
       distribution. */

// $Header: /code/doj/color.hpp,v 1.8 2006/02/20 23:56:21 doj Exp $

#ifndef COLORS__H
#define COLORS__H

#include <iostream>

#ifdef _MSC_VER
typedef unsigned char  uint8_t;
typedef unsigned short uint16_t;
typedef unsigned long  uint32_t;
#else
#include <stdint.h>
#endif

typedef float colorfloattype;

namespace doj
{
  class cie;
  class cmy;
  class cmyk;
  class fsd;
  class gray;
  class hls;
  class hsv;
  class hwb;
  class lab;
  class lch;
  class lchab;
  class lchuv;
  class luv;
  class bns;
  class ncs;
  class rgba;
  class srgb;
  class uvw;
  class xyz;
  class xyZ;
  class ycC;
  class ycrcb;
  class yes;
  class yiq;
  class yuv;

  class rgba
  {
  public:
    colorfloattype r, g, b, a; // [0..1]

    rgba(const colorfloattype rr=0, const colorfloattype gg=0, const colorfloattype bb=0, const colorfloattype aa=0) :
      r(rr),
      g(gg),
      b(bb),
      a(aa)
    { }

    rgba& Init(const colorfloattype rr=0, const colorfloattype gg=0, const colorfloattype bb=0, const colorfloattype aa=0)
    {
      r=rr;
      g=gg;
      b=bb;
      a=aa;
      return *this;
    }

    rgba& operator=(const cmy&);
    rgba& operator=(const cmyk&);
    rgba& operator=(const hsv&);
    rgba& operator=(const hls&);
    rgba& operator=(const hwb&);
    rgba& operator=(const gray&);
    rgba& operator=(const xyz&);
    rgba& operator=(const yuv&);
    rgba& operator=(const yiq&);
    rgba& operator=(const ycrcb&);

    rgba& dreidreizwei(uint8_t);
    rgba& fuenffuenffuenf(uint16_t);
    rgba& fuenfsechsfuenf(uint16_t);
    rgba& argb(uint32_t i);

    std::ostream& dump(std::ostream& os) const;
    std::istream& scan(std::istream& is);

    rgba& operator+=(const rgba& z)
    {
      r+=z.r; g+=z.g; b+=z.b; a+=z.a;
      return *this;
    }

    rgba& operator-=(const rgba& z)
    {
      r-=z.r; g-=z.g; b-=z.b; a-=z.a;
      return *this;
    }

    rgba& operator*=(const colorfloattype f)
    {
      r*=f; g*=f; b*=f; a*=f;
      return *this;
    }

    rgba& invert();
  };

  inline std::ostream& operator<<(std::ostream& os, const rgba& r)
  {
    return r.dump(os);
  }

  inline std::istream& operator>>(std::istream& is, rgba& r)
  {
    return r.scan(is);
  }

  inline rgba operator*(const rgba& r, const colorfloattype f)
  {
    rgba t(r);
    return t*=f;
  }

  inline rgba operator+(const rgba& a, const rgba& b)
  {
    rgba t(a);
    return t+=b;
  }

  inline rgba operator-(const rgba& a, const rgba& b)
  {
    rgba t(a);
    return t-=b;
  }

  class hsv
  {
  public:
    /// hue value [0..360]
    colorfloattype h;
    /// saturation [0..1]
    colorfloattype s;
    /// value [0..1]
    colorfloattype v;

    hsv(const colorfloattype hh=0, const colorfloattype ss=0, const colorfloattype vv=0) :
      h(hh),
      s(ss),
      v(vv)
    { }

    hsv& operator=(const rgba&);

  };

  class hls
  {
  public:
    /// hue [0..1]
    colorfloattype h;
    /// lightness [0..1]
    colorfloattype l;
    /// saturation [0..1]
    colorfloattype s;

    hls(const colorfloattype hh=0, const colorfloattype ll=0, const colorfloattype ss=0) :
      h(hh),
      l(ll),
      s(ss)
    { }

    hls& operator=(const rgba&);

  };

  // hwb is a superior system to hsv and hls
  class hwb
  {
  public:
    colorfloattype h, w, b; // [0..1]

    hwb(const colorfloattype hh=0, const colorfloattype ww=0, const colorfloattype bb=0) :
      h(hh),
      w(ww),
      b(bb)
    { }

    hwb& operator=(const rgba&);

  };


  class cmyk
  {
  public:
    colorfloattype c, m, y, k;

    cmyk(const colorfloattype cc=0, const colorfloattype mm=0, const colorfloattype yy=0, const colorfloattype kk=0) :
      c(cc),
      m(mm),
      y(yy),
      k(kk)
    { }

    cmyk& operator=(const rgba&);
    cmyk& operator=(const cmy&);

  };

  class cmy
  {
  public:
    colorfloattype c, m, y;

    cmy(const colorfloattype cc=0, const colorfloattype mm=0, const colorfloattype yy=0) :
      c(cc),
      m(mm),
      y(yy)
    { }

    cmy& operator=(const rgba&);
    cmy& operator=(const cmyk&);

  };


  class ycrcb
  {
  public:
    colorfloattype y, cr, cb;

    ycrcb(const colorfloattype Y=0, const colorfloattype CR=0, const colorfloattype CB=0) :
      y(Y),
      cr(CR),
      cb(CB)
    { }

    ycrcb& operator=(const rgba&);

  };

  /**
     YUV is like YIQ, except that it is the PAL/European standard.
   */
  class yuv
  {
  public:
    colorfloattype y, u, v;

    yuv(const colorfloattype Y=0, const colorfloattype uu=0, const colorfloattype vv=0) :
      y(Y),
      u(uu),
      v(vv)
    { }

    yuv& operator=(const rgba&);
    yuv& operator=(const ycrcb&);
    yuv& operator=(const yiq&);

  };

  class gray
  {
  public:
    colorfloattype g, a;

    gray(const colorfloattype gg=0, const colorfloattype aa=0) :
      g(gg),
      a(aa)
    { }

    gray& operator=(const rgba&);

  };

  class xyz
  {
  public:
    colorfloattype x, y, z;

    xyz(const colorfloattype xx=0, const colorfloattype yy=0, const colorfloattype zz=0) :
      x(xx),
      y(yy),
      z(zz)
    { }

    xyz& operator=(const rgba&);
    xyz& operator=(const yes&);

  };

  class lch
  {
  public:
    colorfloattype l, c, h;

    lch(const colorfloattype ll=0, const colorfloattype cc=0, const colorfloattype hh=0) :
      l(ll),
      c(cc),
      h(hh)
    { }

    lch& operator=(const lab&);

  };

  class lab
  {
  public:
    colorfloattype l, a, b;

    lab(const colorfloattype ll=0, const colorfloattype aa=0, const colorfloattype bb=0) :
      l(ll),
      a(aa),
      b(bb)
    { }

  };

  /**
     The YIQ system is the colour primary system adopted by NTSC for
     colour television broadcasting. The YIQ color solid is formed by a
     linear transformation of the RGB cube. Its purpose is to exploit
     certain characteristics of the human visual system to maximize the use
     of a fixed bandwidth.
  */
  class yiq
  {
  public:
    colorfloattype y, i, q;

    yiq(const colorfloattype yy=0, const colorfloattype ii=0, const colorfloattype qq=0) :
      y(yy),
      i(ii),
      q(qq)
    { }

    yiq& operator=(const rgba&);
    yiq& operator=(const yuv&);

  };

  class yes
  {
  public:
    colorfloattype y, e, s;

    yes(const colorfloattype yy=0, const colorfloattype ee=0, const colorfloattype ss=0) :
      y(yy),
      e(ee),
      s(ss)
    { }

    yes& operator=(const xyz&);

  };

  void rgb2yuv(const uint8_t r, const uint8_t g, const uint8_t b, uint8_t& y, uint8_t& u, uint8_t v);
  void yuv2rgb(const uint8_t y, const uint8_t u, const uint8_t v, uint8_t& r, uint8_t& g, uint8_t b);
}


#endif

/*
nm2hue() // nanometer to hue
411torgb()
pantone2rgb()
ral2rgb()

xyz2luv()
{
if (Y/Yn > 0.008856)
    L = 116.0 * (Y / Yn)1/3 - 16.0;
else
    L = 903.3 * (Y / Yn);
u = 13.0 * L * (u' - u'n);
v = 13.0 * L * (v' - v'n);

where

u' = 4 * x / (X + 15*Y + 3*Z);
v' = 9 * y / (X + 15*Y + 3*Z);

and u'n, v'n, and Yn are the u' , v' , and Y values for the reference white point.
}


xyz2lab()
{
if (Y/Yn > 0.008856)
    L = 116.0 * (Y / Yn)1/3 - 16.0;
else
    L = 903.3 * (Y / Yn)
a = 500.0 * ( (X / Xn)1/3 - (Y / Yn)1/3 );
b = 500.0 * ( (Y / Yn)1/3 - (Z / Zn)1/3 );

where Xn, Yn, and Zn are the XYZ values for the reference white point.
}

*/
