/* Copyright (c) 2004,2005 Dirk Jagdmann <doj@cubic.org>

This software is provided 'as-is', without any express or implied
warranty. In no event will the authors be held liable for any damages
arising from the use of this software.

Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute it
freely, subject to the following restrictions:

    1. The origin of this software must not be misrepresented; you
       must not claim that you wrote the original software. If you use
       this software in a product, an acknowledgment in the product
       documentation would be appreciated but is not required.

    2. Altered source versions must be plainly marked as such, and
       must not be misrepresented as being the original software.

    3. This notice may not be removed or altered from any source
       distribution. */

// $Header: /code/doj/functions.hpp,v 1.35 2007/09/16 16:50:47 doj Exp $

#ifndef FUNCTIONS__HPP
#define FUNCTIONS__HPP

#include <cassert>
#include <cstdlib>
#include <constant.hpp>
#include <iterator>
#include <limits>
#include <stdint.h>

namespace doj
{

  /// @return r in degrees
  template <typename T>
  inline T DEG(const T& r)
  {
    return r*M_1_PI*180.0;
  }

  /// @return d in rads
  template <typename T>
  inline T RAD(const T& d)
  {
    return d*M_PI/180.0;
  }

  /// @return cotanges of a
  template <typename T>
  inline T cot(const T& a)
  {
    return 1.0/tan(a);
    // return cos(a)/sin(a);
    // return tan(90-a);
  }

  /// @return logarithm of a with base
  template <typename T>
  inline T logbase(const T& a, const T& base)
  {
    return log(a)/log(base);
  }

  /// @return the logarithm base 2 of num
  inline int intlog2(int num)
  {
    if(num<1)
      return -1;
    int l=-1;
    while(num)
      num/=2, ++l;
    return l;
  }

  /**
     if f is a denormalized number it is converted to 0. See also fpclassify(3)
  */
  inline float flushToZero(volatile float f) // the volatile is important. It forces the compiler not to optimize the use of f
  {
    f += 9.8607615E-32f;
    return f - 9.8607615E-32f;
  }

  /**
     compare if two float numbers are equal with arbitrary
     precision. The precision maxUlps gives the numbers of
     representable floating point numbers A and B may be apart, so
     they compare as equal. See also http://www.cygnus-software.com/papers/comparingfloats/comparingfloats.htm

     @param maxUlps distance between A and B as represantable floating point numbers
  */
  inline bool equal(float A, float B, const int maxUlps)
  {
    if(std::fpclassify(A)==FP_INFINITE || std::fpclassify(B)==FP_INFINITE)
      return A == B;

    if(std::fpclassify(A)==FP_NAN || std::fpclassify(B)==FP_NAN)
      return false;

    if(std::fpclassify(A)==FP_SUBNORMAL)
      A=0;
    if(std::fpclassify(B)==FP_SUBNORMAL)
      B=0;

    union {
      int32_t aInt;
      float aFloat;
    };
    aFloat=A;

    // Make aInt lexicographically ordered as a twos-complement int
    if (aInt < 0)
      aInt = std::numeric_limits<int32_t>::min() - aInt;

    union {
      int32_t bInt;
      float bFloat;
    };
    bFloat=B;

    // Make bInt lexicographically ordered as a twos-complement int
    if (bInt < 0)
      bInt = std::numeric_limits<int32_t>::min() - bInt;

    // Now we can compare aInt and bInt to find out how far apart A and B are.
    return (abs(aInt - bInt)<=maxUlps)?true:false;
  }

  /**
     compare if to double numbers are equal with arbitrary
     precision. The precision maxUlps gives the numbers of
     representable floating point numbers A and B may be apart, so
     they compare as equal. See also http://www.cygnus-software.com/papers/comparingfloats/comparingfloats.htm

     @param maxUlps distance between A and B as represantable floating point numbers
  */
  inline bool equal(double A, double B, const int maxUlps)
  {
    if(std::fpclassify(A)==FP_INFINITE || std::fpclassify(B)==FP_INFINITE)
      return A == B;

    if(std::fpclassify(A)==FP_NAN || std::fpclassify(B)==FP_NAN)
      return false;

    if(std::fpclassify(A)==FP_SUBNORMAL)
      A=0;
    if(std::fpclassify(B)==FP_SUBNORMAL)
      B=0;

    union {
      int64_t aInt;
      double aFloat;
    };
    aFloat=A;

    // Make aInt lexicographically ordered as a twos-complement int
    if (aInt < 0)
      aInt = std::numeric_limits<int64_t>::min() - aInt;

    union {
      int64_t bInt;
      double bFloat;
    };
    bFloat=B;

    // Make bInt lexicographically ordered as a twos-complement int
    if (bInt < 0)
      bInt = std::numeric_limits<int64_t>::min() - bInt;

    // Now we can compare aInt and bInt to find out how far apart A and B are.
    return (abs(aInt - bInt)<=maxUlps)?true:false;
  }

  /**
     compares if two floats a,b are equal in a range defined by
     fuzz. fuzz is a scaling factor that should be applied if both
     values are in the range from [0-1]. If a is greater, fuzz is
     scaled accordingly, so the comparision works on the same
     magnitude.

     @return true if a==b in a range defined by fuzz
  */
  inline bool fuzzy_equal(const double a, const double b, const double fuzz=0.000001)
  {
    if(a==b) return true;
    const double aa=fabs(a) + 1;
    return fabs(a-b) <= (isinf(aa) ? fuzz : fuzz*aa);
  }

  inline bool isNaN(const double d)
  {
    return !((d<0.0) || (d>=0.0));
  }

  /// @return x^y
  template <typename T>
  T intpow(T x, int y)
  {
    if(x==static_cast<T>(0))
      return 0;
    if(y<0)
      return static_cast<T>(1)/intpow(x, -y);

    if(y==0)
      return 1;
    if(y==1)
      return x;
    if(y==2)
      return x*x;

    T a=1;
    while(y)
      {
	if(y&1)
	  a*=x;
	x*=x;
	y>>=1;
      }

    return a;
  }

  /**
     auf deutsch: groesster gemeinsamer teiler ggt
     @return greatest common divisor of x and y
  */
  template <typename T>
  T gcd(T x, T y)
  {
    while(x!=y)
      if(x>y)
	x-=y;
      else
	y-=x;
    return x;
  }

  /**
     auf deutsch: kleinstes gemeinsames vielfaches kgv
     @return the least common multiple of x and y
  */
  template <typename T>
  T lcm(T x, T y)
  {
    return x*y/gcd(x,y);
  }

  /**
     @return f in the range [min..max]
  */
  template <typename T>
  inline T clamp(const T& f, const T& min=0, const T& max=1)
  {
    if(f<min)
      return min;
    if(f>max)
      return max;
    return f;
  }

  /** Total number of combinations of n elements taken r elements at a
  time.  Uses an adaptation of the following formula: nCr = n! /
  (n-r)!r!

  @author John Dibling
  */
  template <typename T>
  T numberOfCombinations(const T& n, const T& r)
  {
    T i, j, numer=1, denom=1;
    // compute numerator
    for( i = n, j = (n-r)>r?n-r:r; i > j; --i )
      numer *= i;
    // compute denominator
    for( i = n-r<r?n-r:r; i > 1; --i )
      denom *= i;
    // epilogue
    return numer/denom;
  }

  /** Total number of permutations of n elements taken r elements at a
  time.  Uses an adaptation of the following formula: nPr = n! /
  (n-r)!

  @author John Dibling
  */
  template <typename T>
  T numberOfPermutations(T n, T r)
  {
    T result = 1;
    for(T j = n-r; n > j; --n )
      result *= n;
    return result;
  }

  /**
     calculate decibels according to db=20.0*log10(value / reference)
  */
  template <typename T>
  inline T db(const T& val, const T& ref=1)
  {
    assert(ref != static_cast<T>(0));
    const T d=val/ref;
    if(d <= static_cast<T>(0))
      return std::numeric_limits<T>::min();
    return static_cast<T>(20)*log10(d);
  }

  /**
     calculates the decibel value of the maximum/minimum element from [begin,end). See db()
  */
  template <typename _Iter>
  inline typename std::iterator_traits<_Iter>::value_type db_buf(_Iter begin, const _Iter end, const typename std::iterator_traits<_Iter>::value_type ref=1)
  {
    typedef typename std::iterator_traits<_Iter>::value_type T;
    T m(std::numeric_limits<T>::epsilon());
    while(begin!=end)
      m=std::max(std::fabs(*begin++), m);
    return db(m, ref);
  }

  /// @return if i==2^x
  inline bool isPowerOf2(unsigned i)
  {
    return i > 0 && (i & (i - 1)) == 0;
  }

  /// @return +1 if arg>=0 else -1
  template <typename T>
  T sign(const T arg)
  {
    return (arg>=0)?+1:-1;
  }

  namespace {

    // TODO: dct krams auf nicht quadratische matrizen erweitern

    inline double c(const unsigned i, const unsigned j, const unsigned N)
    {
      return std::sqrt(((i==0)?1.0:2.0)/N) * std::sqrt(((j==0)?1.0:2.0)/N);
    }

    template <typename Matrix, typename Element>
    Matrix dct_helper(const Matrix& s, const unsigned N, const Element&)
    {
      Matrix t(s);
      if(N==0)
	return t;

      const Element fac=1.0/(2.0*N);
      for(unsigned i=0; i!=N; ++i)
	for(unsigned j=0; j!=N; ++j)
	  {
	    Element sum=0;
	    for(unsigned n=0; n!=N; ++n)
	      for(unsigned m=0; m!=N; ++m)
		sum+=s[n][m] * std::cos(M_PI*(2*m+1)*j*fac) * std::cos(M_PI*(2*n+1)*i*fac);
	    t[i][j] = sum * c(i,j,N);
	  }

      return t;
    }

    template <typename Matrix, typename Element>
    Matrix idct_helper(const Matrix& s, const unsigned N, const Element&)
    {
      Matrix t(s);
      if(N==0)
	return t;

      const Element fac=1.0/(2.0*N);
      for(unsigned n=0; n!=N; ++n)
	for(unsigned m=0; m!=N; ++m)
	  {
	    Element sum=0;
	    for(unsigned i=0; i!=N; ++i)
	      for(unsigned j=0; j!=N; ++j)
		sum+=s[i][j] * c(i,j,N) * std::cos(M_PI*(2*m+1)*j*fac) * std::cos(M_PI*(2*n+1)*i*fac);
	    t[n][m] = sum;
	  }

      return t;
    }
  }

  /**
     compute the discrete cosinus transform of s. s has to be a square
     matrix of size N. The elements of s must be accessible with
     s[y][x].

     @return the cosine transformed matrix of the same dimensions as s
  */
  template <typename Matrix>
  Matrix dct(const Matrix& s, const unsigned N)
  {
    return dct_helper(s,N,s[0][0]);
  }

  /**
     compute the inverse discrete cosinus transform of s. s has to be
     a square matrix of size N. The elements of s must be accessible
     with s[y][x].

     @return the inverse cosine transformed matrix of the same dimensions as s
  */
  template <typename Matrix>
  Matrix idct(const Matrix& s, const unsigned N)
  {
    return idct_helper(s,N,s[0][0]);
  }

}

#endif
