/* Copyright (c) 2004 Dirk Jagdmann <doj@cubic.org>

This software is provided 'as-is', without any express or implied
warranty. In no event will the authors be held liable for any damages
arising from the use of this software.

Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute it
freely, subject to the following restrictions:

    1. The origin of this software must not be misrepresented; you
       must not claim that you wrote the original software. If you use
       this software in a product, an acknowledgment in the product
       documentation would be appreciated but is not required.

    2. Altered source versions must be plainly marked as such, and
       must not be misrepresented as being the original software.

    3. This notice may not be removed or altered from any source
       distribution. */

// $Header: /code/doj/gl.hpp,v 1.11 2006/07/18 18:48:57 doj Exp $

#ifndef GL__HPP
#define GL__HPP

#include <matrix.hpp>
#include <vektor.hpp>
#include <functions.hpp>

#ifdef _MSC_VER
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <wingdi.h>
#endif

#include <GL/gl.h>
#include <GL/glu.h>

#include <iostream>

namespace doj
{

  /// call glNormal3f() with v
  template <typename T>
  inline void glNormal(Vektor<T> v)
  {
    v.normalise();
    glNormal3f(v.x, v.y, v.z);
  }

  /// call glVertex4f() with v
  template <typename T>
  inline void glVertex(const Vektor<T>& v)
  {
    glVertex4f(v.x, v.y, v.z, v.w);
  }

  /// call glTexCoord2f() with v
  template <typename T>
  inline void glTexCoord(const Vektor<T>& v)
  {
    glTexCoord2f(v.x, v.y);
  }

  /// call glTranslatef with v
  template <typename T>
  inline void glTranslate(const Vektor<T>& v)
  {
    glTranslatef(v.x, v.y, v.z);
  }

  /// call glScalef with v
  template <typename T>
  inline void glScale(const Vektor<T>& v)
  {
    glScalef(v.x, v.y, v.z);
  }

  /// call glRotatef with angle a and Vektor v
  template <typename T>
  inline void glRotate(const float a, const Vektor<T>& v)
  {
    glRotatef(a, v.x, v.y, v.z);
  }

  /// perform uniform scaling with glScalef()
  inline void glScale(const float f)
  {
    glScalef(f,f,f);
  }


  inline void gluOrthoBefore(const float width, const float height)
  {
    glMatrixMode(GL_PROJECTION);
    glPushMatrix();
    glLoadIdentity();
    gluPerspective(90, width/height, 0.1, 1.1);

    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glLoadIdentity();
    const float prop=height/width;
    gluOrtho2D(-prop, +prop, -1, 1);
  }

  inline void gluOrthoAfter()
  {
    glMatrixMode(GL_PROJECTION);
    glPopMatrix();
    glMatrixMode(GL_MODELVIEW);
    glPopMatrix();
  }

  /// initialize the view to pixel=texel := 1:1
  inline void glMap1to1(const int width, const int height)
  {
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    glOrtho(0, 1, 0, 1, 0, 1);
    glViewport(0,0,width,height);
  }


  template <typename T>
  inline void glMultMatrix(const T* m)
  {
#ifdef DOJDEBUG
    std::clog << "doj::glMultMatrix(T) called ! this is suboptimal" << std::endl;
#endif
    double M[16] = {
      m[0], m[1], m[2], m[3],
      m[4], m[5], m[6], m[7],
      m[8], m[9], m[10],m[11],
      m[12],m[13],m[14],m[15]
    };
    glMultMatrixd(M);
  }

  template <>
  inline void glMultMatrix(const double *m)
  {
    glMultMatrixd(m);
  }

  template <>
  inline void glMultMatrix(const float *m)
  {
    glMultMatrixf(m);
  }

  template <typename T>
  inline void glMultMatrix(const Matrix<T>& m)
  {
    glMultMatrix(m.array());
  }

  // ripped from http://occs.cs.oberlin.edu/faculty/jdonalds/357/lecture23.html
  template <typename T, typename L>
  inline void shadowPlaneProjection(const Vektor<L>& light, const T A, const T B, const T C, T D)
  {
    D += A*light.x + B*light.y + C*light.z;
    if(D!=0.0)
      {
	// remember! OpenGL matrix operations are in reverse order!
	glTranslate(light);

	T P[16]={
	  1,0,0,-A/(.999*D),
	  0,1,0,-B/(.999*D),
	  0,0,1,-C/(.999*D),
	  0,0,0,0
	};
	glMultMatrix(P);

	glTranslate(-light);
      }
  }

  template <typename T, typename S>
  inline void shadowPlaneProjection(const Vektor<T>& light, const Vektor<S>& planeNormal, const Vektor<S>& planePoint)
  {
    shadowPlaneProjection(light, planeNormal.x, planeNormal.y, planeNormal.z, planeNormal.dot(planePoint));
  }

#if 0
  // TODO: don't know what second parameter (pl) should be?
  template <typename T, typename S>
  inline void shadowPlaneProjection(const Vektor<T>& light, const Plane<S>& pl)
  {
    shadowPlaneProjection(light, pl.n, pl.p);
  }
#endif
}

#endif
