/* Copyright (c) 2004 Dirk Jagdmann <doj@cubic.org>

This software is provided 'as-is', without any express or implied
warranty. In no event will the authors be held liable for any damages
arising from the use of this software.

Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute it
freely, subject to the following restrictions:

    1. The origin of this software must not be misrepresented; you
       must not claim that you wrote the original software. If you use
       this software in a product, an acknowledgment in the product
       documentation would be appreciated but is not required.

    2. Altered source versions must be plainly marked as such, and
       must not be misrepresented as being the original software.

    3. This notice may not be removed or altered from any source
       distribution. */

// $Header: /code/doj/matrix.hpp,v 1.7 2005/02/26 19:31:34 doj Exp $

#ifndef MATRIX__HPP
#define MATRIX__HPP

#include <iostream>
#include <string>

#include <functions.hpp>
#include <vektor.hpp>

namespace doj
{
  /**
     Matrix provides a 4x4 matrix of arbitrary type. It is intended for
     3D calculations and OpenGL.
  */
  template<typename T>
  class Matrix
  {
  protected:
    int repaircount;
    bool identitymatrix_;
    bool modified_;

    /** the matrix values are stored in this array. The array is
	organized in row mode allowing easy use with OpenGL.
    */
    T a[16];

  public:

    Matrix() :
      repaircount(0),
      identitymatrix_(false),
      modified_(true)
    {
    }

    Matrix<T>& operator=(const Matrix<T>& m)
    {
      if(this==&m)
	return *this;

      if(identitymatrix_ && m.identitymatrix_)
	return *this;

      identitymatrix_=m.identitymatrix_;
      for(int i=0; i<16; i++)
	a[i]=m.a[i];

      modified_=true;
      return *this;
    }

    /**
       set this to the identity matrix, with each element on the main
       diagonal set to 1.
     */
    Matrix<T>& identity()
    {
      if(identitymatrix_)
	return *this;

      identitymatrix_=true;
      a[ 0]=1; a[ 4]=0; a[ 8]=0; a[12]=0;
      a[ 1]=0; a[ 5]=1; a[ 9]=0; a[13]=0;
      a[ 2]=0; a[ 6]=0; a[10]=1; a[14]=0;
      a[ 3]=0; a[ 7]=0; a[11]=0; a[15]=1;

      modified_=true;
      return *this;
    }

    /**
       same as identity()
    */
    Matrix<T>& Init()
    {
      return identity();
    }

    /**
       @return true if this is an identity Matrix. false if this is an
       arbitrary Matrix.
    */
    bool identitymatrix() const
    {
      return identitymatrix_;
    }

    /**
       @return if the matrix was modified.
    */
    bool modified() const
    {
      return modified_;
    }

    const T* array() const
    {
      return a;
    }

    /**
       transpose this. Element [x,y] becomes [y,x].
    */
    Matrix<T>& transpose()
    {
      if(identitymatrix_)
	return *this;

      for(int y=0; y<4; y++)
	for(int x=y+1; x<4; x++)
	  {
	    const T t=a[y*4+x];
	    a[y*4+x]=a[x*4+y];
	    a[x*4+y]=t;
	  }

      modified_=true;
      return *this;
    }


    /**
       substitute this by inverse this

       @return true if inverse was caluclated. false if matrix is singular
       matrix, thus no inverse matrix exists.
    */
    bool invert()
    {
      if(identitymatrix_)
	return true;

      // ripped from Mesa3D, but they ripped it from Jacques Leroy <jle@star.be>

#define SWAP_ROWS(a, b) { T *_tmp = a; (a)=(b); (b)=_tmp; }
#define MAT(m,r,c) (m)[(c)*4+(r)]

      T wtmp[4][8];
      T m0, m1, m2, m3, s;
      T *r0 = wtmp[0], *r1 = wtmp[1], *r2 = wtmp[2], *r3 = wtmp[3];

      r0[0] = MAT(a,0,0);
      r0[1] = MAT(a,0,1);
      r0[2] = MAT(a,0,2);
      r0[3] = MAT(a,0,3);
      r0[4] = 1.0;
      r0[5] = r0[6] = r0[7] = 0.0;

      r1[0] = MAT(a,1,0);
      r1[1] = MAT(a,1,1);
      r1[2] = MAT(a,1,2);
      r1[3] = MAT(a,1,3);
      r1[5] = 1.0;
      r1[4] = r1[6] = r1[7] = 0.0;

      r2[0] = MAT(a,2,0);
      r2[1] = MAT(a,2,1);
      r2[2] = MAT(a,2,2);
      r2[3] = MAT(a,2,3);
      r2[6] = 1.0;
      r2[4] = r2[5] = r2[7] = 0.0;

      r3[0] = MAT(a,3,0);
      r3[1] = MAT(a,3,1);
      r3[2] = MAT(a,3,2);
      r3[3] = MAT(a,3,3);
      r3[7] = 1.0;
      r3[4] = r3[5] = r3[6] = 0.0;

      // choose pivot - or die
      if (fabs(r3[0])>fabs(r2[0])) SWAP_ROWS(r3, r2);
      if (fabs(r2[0])>fabs(r1[0])) SWAP_ROWS(r2, r1);
      if (fabs(r1[0])>fabs(r0[0])) SWAP_ROWS(r1, r0);
      if (0.0 == r0[0])
	return false;

      // eliminate first variable
      m1 = r1[0]/r0[0]; m2 = r2[0]/r0[0]; m3 = r3[0]/r0[0];
      s = r0[1]; r1[1] -= m1 * s; r2[1] -= m2 * s; r3[1] -= m3 * s;
      s = r0[2]; r1[2] -= m1 * s; r2[2] -= m2 * s; r3[2] -= m3 * s;
      s = r0[3]; r1[3] -= m1 * s; r2[3] -= m2 * s; r3[3] -= m3 * s;
      s = r0[4];
      if (s != 0.0) { r1[4] -= m1 * s; r2[4] -= m2 * s; r3[4] -= m3 * s; }
      s = r0[5];
      if (s != 0.0) { r1[5] -= m1 * s; r2[5] -= m2 * s; r3[5] -= m3 * s; }
      s = r0[6];
      if (s != 0.0) { r1[6] -= m1 * s; r2[6] -= m2 * s; r3[6] -= m3 * s; }
      s = r0[7];
      if (s != 0.0) { r1[7] -= m1 * s; r2[7] -= m2 * s; r3[7] -= m3 * s; }

      // choose pivot - or die
      if (fabs(r3[1])>fabs(r2[1])) SWAP_ROWS(r3, r2);
      if (fabs(r2[1])>fabs(r1[1])) SWAP_ROWS(r2, r1);
      if (0.0 == r1[1])
	return false;

      // eliminate second variable
      m2 = r2[1]/r1[1]; m3 = r3[1]/r1[1];
      r2[2] -= m2 * r1[2]; r3[2] -= m3 * r1[2];
      r2[3] -= m2 * r1[3]; r3[3] -= m3 * r1[3];
      s = r1[4]; if (0.0 != s) { r2[4] -= m2 * s; r3[4] -= m3 * s; }
      s = r1[5]; if (0.0 != s) { r2[5] -= m2 * s; r3[5] -= m3 * s; }
      s = r1[6]; if (0.0 != s) { r2[6] -= m2 * s; r3[6] -= m3 * s; }
      s = r1[7]; if (0.0 != s) { r2[7] -= m2 * s; r3[7] -= m3 * s; }

      // choose pivot - or die
      if (fabs(r3[2])>fabs(r2[2])) SWAP_ROWS(r3, r2);
      if (0.0 == r2[2])
	return false;

      // eliminate third variable
      m3 = r3[2]/r2[2];
      r3[3] -= m3 * r2[3], r3[4] -= m3 * r2[4];
      r3[5] -= m3 * r2[5], r3[6] -= m3 * r2[6];
      r3[7] -= m3 * r2[7];

      // last check
      if (0.0 == r3[3])
	return false;

      s = 1.0F/r3[3];

      // now back substitute row 3
      r3[4] *= s; r3[5] *= s; r3[6] *= s; r3[7] *= s;    m2 = r2[3];

      // now back substitute row 2
      s  = 1.0F/r2[2];
      r2[4] = s * (r2[4] - r3[4] * m2); r2[5] = s * (r2[5] - r3[5] * m2);
      r2[6] = s * (r2[6] - r3[6] * m2); r2[7] = s * (r2[7] - r3[7] * m2);
      m1 = r1[3];
      r1[4] -= r3[4] * m1, r1[5] -= r3[5] * m1;
      r1[6] -= r3[6] * m1, r1[7] -= r3[7] * m1;
      m0 = r0[3];
      r0[4] -= r3[4] * m0, r0[5] -= r3[5] * m0;
      r0[6] -= r3[6] * m0, r0[7] -= r3[7] * m0;    m1 = r1[2];

      // now back substitute row 1
      s  = 1.0F/r1[1];
      r1[4] = s * (r1[4] - r2[4] * m1), r1[5] = s * (r1[5] - r2[5] * m1);
      r1[6] = s * (r1[6] - r2[6] * m1), r1[7] = s * (r1[7] - r2[7] * m1);
      m0 = r0[2];
      r0[4] -= r2[4] * m0, r0[5] -= r2[5] * m0;
      r0[6] -= r2[6] * m0, r0[7] -= r2[7] * m0;    m0 = r0[1];

      // now back substitute row 0
      s  = 1.0F/r0[0];
      r0[4] = s * (r0[4] - r1[4] * m0), r0[5] = s * (r0[5] - r1[5] * m0);
      r0[6] = s * (r0[6] - r1[6] * m0), r0[7] = s * (r0[7] - r1[7] * m0);

      MAT(a,0,0) = r0[4]; MAT(a,0,1) = r0[5];
      MAT(a,0,2) = r0[6]; MAT(a,0,3) = r0[7];
      MAT(a,1,0) = r1[4]; MAT(a,1,1) = r1[5];
      MAT(a,1,2) = r1[6]; MAT(a,1,3) = r1[7];
      MAT(a,2,0) = r2[4]; MAT(a,2,1) = r2[5];
      MAT(a,2,2) = r2[6]; MAT(a,2,3) = r2[7];
      MAT(a,3,0) = r3[4]; MAT(a,3,1) = r3[5];
      MAT(a,3,2) = r3[6]; MAT(a,3,3) = r3[7];

      modified_=true;
      return true;

#undef SWAP_ROWS
    }

    bool invert3dMatrix()
    {
      if(identitymatrix_)
	return true;

      // Calculate the determinant of upper left 3x3 submatrix and determine if the matrix is singular.
      T pos=0, neg=0;
      T t = MAT(a,0,0) * MAT(a,1,1) * MAT(a,2,2);
      if (t >= 0.0) pos += t; else neg += t;    t =  MAT(a,1,0) * MAT(a,2,1) * MAT(a,0,2);
      if (t >= 0.0) pos += t; else neg += t;    t =  MAT(a,2,0) * MAT(a,0,1) * MAT(a,1,2);
      if (t >= 0.0) pos += t; else neg += t;    t = -MAT(a,2,0) * MAT(a,1,1) * MAT(a,0,2);
      if (t >= 0.0) pos += t; else neg += t;    t = -MAT(a,1,0) * MAT(a,0,1) * MAT(a,2,2);
      if (t >= 0.0) pos += t; else neg += t;    t = -MAT(a,0,0) * MAT(a,2,1) * MAT(a,1,2);
      if (t >= 0.0) pos += t; else neg += t;

      T det = pos + neg;
      if (det*det < 1e-25)
	return false;

      det = 1.0 / det;

      Matrix<T> n;
      T *out = n.a;
      MAT(out,0,0) = (  (MAT(a,1,1)*MAT(a,2,2) - MAT(a,2,1)*MAT(a,1,2) )*det);
      MAT(out,0,1) = (- (MAT(a,0,1)*MAT(a,2,2) - MAT(a,2,1)*MAT(a,0,2) )*det);
      MAT(out,0,2) = (  (MAT(a,0,1)*MAT(a,1,2) - MAT(a,1,1)*MAT(a,0,2) )*det);
      MAT(out,1,0) = (- (MAT(a,1,0)*MAT(a,2,2) - MAT(a,2,0)*MAT(a,1,2) )*det);
      MAT(out,1,1) = (  (MAT(a,0,0)*MAT(a,2,2) - MAT(a,2,0)*MAT(a,0,2) )*det);
      MAT(out,1,2) = (- (MAT(a,0,0)*MAT(a,1,2) - MAT(a,1,0)*MAT(a,0,2) )*det);
      MAT(out,2,0) = (  (MAT(a,1,0)*MAT(a,2,1) - MAT(a,2,0)*MAT(a,1,1) )*det);
      MAT(out,2,1) = (- (MAT(a,0,0)*MAT(a,2,1) - MAT(a,2,0)*MAT(a,0,1) )*det);
      MAT(out,2,2) = (  (MAT(a,0,0)*MAT(a,1,1) - MAT(a,1,0)*MAT(a,0,1) )*det);

      // Do the translation part
      MAT(out,0,3) = - (MAT(a,0,3) * MAT(out,0,0) +
			MAT(a,1,3) * MAT(out,0,1) +
			MAT(a,2,3) * MAT(out,0,2) );
      MAT(out,1,3) = - (MAT(a,0,3) * MAT(out,1,0) +
			MAT(a,1,3) * MAT(out,1,1) +
			MAT(a,2,3) * MAT(out,1,2) );
      MAT(out,2,3) = - (MAT(a,0,3) * MAT(out,2,0) +
			MAT(a,1,3) * MAT(out,2,1) +
			MAT(a,2,3) * MAT(out,2,2) );

      *this=n;
      return true;
    }

#undef MAT


    /**
       transform this by a rotation of angle phi around a vector
       (x_,y_,z_).
       @param phi angle in degrees
    */
    Matrix<T>& rotate(const T& phi, const T& x_, const T& y_, const T& z_)
    {
      const T len=x_*x_ + y_*y_ + z_*z_;
      if (len <= 1.0e-4)
	return *this;

      const T norm = 1.0/sqrt(len);

      const T x = x_*norm;
      const T y = y_*norm;
      const T z = z_*norm;

      const T x2 = x * x;
      const T y2 = y * y;
      const T z2 = z * z;
      const T xy = x * y;
      const T yz = y * z;
      const T zx = z * x;

      const T s = sin(RAD(phi));
      const T c = cos(RAD(phi));

      const T xs = x * s;
      const T ys = y * s;
      const T zs = z * s;

      const T c1 = 1.0 - c;

      Matrix<T> m;
      m[ 0]=x2*c1+c;  m[ 4]=xy*c1-zs; m[ 8]=zx*c1+ys; m[12]=0;
      m[ 1]=xy*c1+zs; m[ 5]=y2*c1+c;  m[ 9]=yz*c1-xs; m[13]=0;
      m[ 2]=zx*c1-ys; m[ 6]=yz*c1+xs; m[10]=z2*c1+c;  m[14]=0;
      m[ 3]=0;        m[ 7]=0;        m[11]=0;        m[15]=1;

      return *this *= m;
    }

    /**
       transform this by a rotation of angle phi around Vektor v.
       @param phi angle in degrees
    */
    Matrix<T>& rotate(const T& phi, const Vektor<T>& v)
    {
      return rotate(phi, v.x, v.y, v.z);
    }

    /**
       transform this by a rotation of angle phi around x axis.
       @param phi angle in degrees
    */
    Matrix<T>& rotateX(const T& phi)
    {
      if(phi==0.0)
	return *this;

      const T s=sin(RAD(phi));
      const T c=cos(RAD(phi));
      Matrix<T> m;
      m[ 0]=1; m[ 4]=0; m[ 8]=0; m[12]=0;
      m[ 1]=0; m[ 5]=c; m[ 9]=-s;m[13]=0;
      m[ 2]=0; m[ 6]=s; m[10]=c; m[14]=0;
      m[ 3]=0; m[ 7]=0; m[11]=0; m[15]=1;
      return *this *= m;
    }

    /**
       transform this by a rotation of angle phi around y axis.
       @param phi angle in degrees
    */
    Matrix<T>& rotateY(const T& phi)
    {
      if(phi==0.0)
	return *this;

      const T s=sin(RAD(phi));
      const T c=cos(RAD(phi));
      Matrix<T> m;
      m[ 0]=c; m[ 4]=0; m[ 8]=s; m[12]=0;
      m[ 1]=0; m[ 5]=1; m[ 9]=0; m[13]=0;
      m[ 2]=-s;m[ 6]=0; m[10]=c; m[14]=0;
      m[ 3]=0; m[ 7]=0; m[11]=0; m[15]=1;
      return *this *= m;
    }

    /**
       transform this by a rotation of angle phi around z axis.
       @param phi angle in degrees
    */
    Matrix<T>& rotateZ(const T& phi)
    {
      if(phi==0.0)
	return *this;

      const T s=sin(RAD(phi));
      const T c=cos(RAD(phi));
      Matrix<T> m;
      m[ 0]=c; m[ 4]=-s;m[ 8]=0; m[12]=0;
      m[ 1]=s; m[ 5]=c; m[ 9]=0; m[13]=0;
      m[ 2]=0; m[ 6]=0; m[10]=1; m[14]=0;
      m[ 3]=0; m[ 7]=0; m[11]=0; m[15]=1;
      return *this *= m;
    }

    /**
       transform this by a view frustum.
       @param l coordinate for left clipping plane
       @param r coordinate for right clipping plane
       @param b coordinate for bottom clipping plane
       @param t coordinate for top clipping plane
       @param n coordinate for near clipping plane
       @param f coordinate for far clipping plane
    */
    Matrix<T>& frustum(const T& l, const T& r, const T& b, const T& t, const T& n, const T& f)
    {
      // ripped from Mesa3D
      Matrix<T> m;
      m[ 0]=2*n/(r-l); m[ 4]=0;         m[ 8]=(r+l)/(r-l);  m[12]=0;
      m[ 1]=0;         m[ 5]=2*n/(t-b); m[ 9]=(t+b)/(t-b);  m[13]=0;
      m[ 2]=0;         m[ 6]=0;         m[10]=-(f+n)/(f-n); m[14]=-2*f*n/(f-n);
      m[ 3]=0;         m[ 7]=0;         m[11]=-1;           m[15]=0;
      return *this *= m;
    }

    /**
       transform this by a orthonormal transformation. This is a parallel projection.
       @param l coordinate for left clipping plane
       @param r coordinate for right clipping plane
       @param b coordinate for bottom clipping plane
       @param t coordinate for top clipping plane
       @param n coordinate for near clipping plane
       @param f coordinate for far clipping plane
    */
    Matrix<T>& ortho(const T& l, const T& r, const T& b, const T& t, const T& n, const T& f)
    {
      // ripped from Mesa3D
      Matrix<T> m;
      m[ 0]=2/(r-l); m[ 4]=0;       m[ 8]=0;        m[12]=-(r+l)/(r-l);
      m[ 1]=0;       m[ 5]=2/(t-b); m[ 9]=0;        m[13]=-(t+b)/(t-b);
      m[ 2]=0;       m[ 6]=0;       m[10]=-2/(f-n); m[14]=-(f+n)/(f-n);
      m[ 3]=0;       m[ 7]=0;       m[11]=0;        m[15]=1;
      return *this *= m;
    }

    /**
       transform this by a two-dimensional orthographic viewing region.
       @param l coordinate for left clipping plane
       @param r coordinate for right clipping plane
       @param b coordinate for bottom clipping plane
       @param t coordinate for top clipping plane
    */
    Matrix<T>& ortho2d(const T& l, const T& r, const T& b, const T& t)
    {
      return ortho(l,r,b,t,-1,1);
    }

    /**
       see gluPerspective(3g)
       @param fov Field-Of-View in degrees
       @param a aspect ratio of screen = width/height
       @param n near clipping plane
       @param f far clipping plane
    */
    Matrix<T>& perspective(const T& fov, const T& a, const T& n, const T& f)
    {
      // ripped from Mesa3D
      const T ymax = n * tan(fov * M_PI / 360.0);
      const T ymin = -ymax;
      const T xmin = ymin * a;
      const T xmax = ymax * a;
      return frustum(xmin, xmax, ymin, ymax, n, f);
    }

    /**
       translate this by (x,y,z).
    */
    Matrix<T>& translate(const T& x, const T& y, const T& z)
    {
      if(identitymatrix_)
	{
	  a[12]=x; a[13]=y; a[14]=z;
	  identitymatrix_=false;
	  modified_=true;
	  return *this;
	}

      a[12] += a[0] * x + a[4] * y + a[8]  * z;
      a[13] += a[1] * x + a[5] * y + a[9]  * z;
      a[14] += a[2] * x + a[6] * y + a[10] * z;

      modified_=true;
      return *this;
    }

    /**
       translate this by t
    */
    Matrix<T>& translate(const Vektor<T>& t)
    {
      return translate(t.x, t.y, t.z);
    }

    /**
       @return the position vector of this.
    */
    Vektor<T> position() const
    {
      return Vektor<T>(a[12], a[13], a[14], 1);
    }

    /**
       @return the right vector of this.
    */
    Vektor<T> right() const
    {
      return Vektor<T>(a[0], a[4], a[8]);
    }

    /**
       @return the up vector of this.
    */
    Vektor<T> up() const
    {
      return Vektor<T>(a[1], a[5], a[9]);
    }

    /**
       @return the forward vector of this.
    */
    Vektor<T> forward() const
    {
      return Vektor<T>(a[2], a[6], a[10]);
    }

    /**
       scale this by (x,y,z)
    */
    Matrix<T>& scale(const T& x, const T& y, const T& z)
    {
#if 0
      Matrix<T> m;
      m[ 0]=x; m[ 4]=0; m[ 8]=0; m[12]=0;
      m[ 1]=0; m[ 5]=y; m[ 9]=0; m[13]=0;
      m[ 2]=0; m[ 6]=0; m[10]=z; m[14]=0;
      m[ 3]=0; m[ 7]=0; m[11]=0; m[15]=1;
      m.identitymatrix_=false;
      return *this *= m;
#else
      if(identitymatrix_)
	{
	  a[0]*=x; a[5]*=y; a[10]*=z;
	  identitymatrix_=false;
	}
      else
	{
	  a[ 0]*=x; a[ 4]*=y; a[ 8]*=z;
	  a[ 1]*=x; a[ 5]*=y; a[ 9]*=z;
	  a[ 2]*=x; a[ 6]*=y; a[10]*=z;
	}
      modified_=true;
      return *this;
#endif
    }

    /**
       scale this by v
    */
    Matrix<T>& scale(const Vektor<T>& v)
    {
      return scale(v.x, v.y, v.z);
    }

    /**
       this is assumed to be a matrix containing a rotation matrix
       along with a translation.  repair the rotation matrix by
       recalculating one rotation vector from the other zwo.
    */
    Matrix<T>& repair()
    {
      if(identitymatrix_)
	return *this;

      identitymatrix_=false;

      Vektor<T> i, j, k;
      switch(++repaircount%3)
	{
	case 0:
	  i.Init(a[ 0], a[ 1], a[ 2]); i.normalise();
	  j.Init(a[ 4], a[ 5], a[ 6]); j.normalise();
	  k=i.cross(j);
	  break;
	case 1:
	  i.Init(a[ 0], a[ 1], a[ 2]); i.normalise();
	  k.Init(a[ 8], a[ 9], a[10]); k.normalise();
	  j=k.cross(i);
	  break;
	case 2:
	  j.Init(a[ 4], a[ 5], a[ 6]); j.normalise();
	  k.Init(a[ 8], a[ 9], a[10]); k.normalise();
	  i=j.cross(k);
	  break;
	}
      a[ 0]=i.x; a[ 4]=j.x; a[ 8]=k.x;
      a[ 1]=i.y; a[ 5]=j.y; a[ 9]=k.y;
      a[ 2]=i.z; a[ 6]=j.z; a[10]=k.z;
      modified_=true;
      return *this;
    }

    /**
       transform this by a lookat Matrix.
       @param eye position of camera
       @param center center point of view plane
       @param up up vector of camera
    */
    Matrix<T>& lookat(const Vektor<T>& eye, const Vektor<T>& center, Vektor<T> up)
    {
      // ripped from gluLookAt(3g)
      Vektor<T> f(eye, center);
      f.normalise();
      Vektor<T> s=f.cross(up);
      up=s.cross(f);
      s.normalise();
      up.normalise();

      Matrix<T> n;
      n.a[ 0]=s.x;  n.a[ 4]=s.y;  n.a[ 8]=s.z;  n.a[12]=0;
      n.a[ 1]=up.x; n.a[ 5]=up.y; n.a[ 9]=up.z; n.a[13]=0;
      n.a[ 2]=-f.x; n.a[ 6]=-f.y; n.a[10]=-f.z; n.a[14]=0;
      n.a[ 3]=0;    n.a[ 7]=0;    n.a[11]=0;    n.a[15]=1;

      *this*=n;
      translate(-eye);

      return *this;
    }

    /**
       create a GLU pick matrix
    */
    Matrix<T>& pick(const T x, const T y, const T width, const T height, int viewport[4])
    {
      // ripped from Mesa3D
      const T sx = viewport[2] / width;
      const T sy = viewport[3] / height;
      const T tx = (viewport[2] + 2.0 * (viewport[0] - x)) / width;
      const T ty = (viewport[3] + 2.0 * (viewport[1] - (viewport[3]-y))) / height;

      Matrix<T> n;
      n.a[ 0]=sx;n.a[ 4]=0; n.a[ 8]=0; n.a[12]=tx;
      n.a[ 1]=0; n.a[ 5]=sy;n.a[ 9]=0; n.a[13]=ty;
      n.a[ 2]=0; n.a[ 6]=0; n.a[10]=1; n.a[14]=0;
      n.a[ 3]=0; n.a[ 7]=0; n.a[11]=0; n.a[15]=1;
      return *this*=n;
    }

    /**
       construct a matrix which projects objects onto plane (A,B,C,D) from lightsource (lightx, lighty, lightz).
       ripped from http://occs.cs.oberlin.edu/faculty/jdonalds/357/lecture23.html
    */
    Matrix<T>& shadowProjection(const T lightx, const T lighty, const T lightz,
				const T A, const T B, const T C, T D)
    {
      D += A*lightx + B*lighty + C*lightz;
      if(D!=0.0)
	{
	  translate(lightx, lighty, lightz);

	  Matrix<T> m;
	  m[ 0]=1; m[ 4]=0; m[ 8]=0; m[12]=0;
	  m[ 1]=0; m[ 5]=1; m[ 9]=0; m[13]=0;
	  m[ 2]=0; m[ 6]=0; m[10]=1; m[14]=0;
	  m[ 3]=-A/(.999*D); m[ 7]=-B/(.999*D); m[11]=-C/(.999*D); m[15]=0;
	  *this*=m;

	  translate(-lightx, -lighty, -lightz);
	}

      return *this;
    }

    /**
       see previous function
    */
    Matrix<T>& shadowProjection(const Vektor<T>& light, const T A, const T B, const T C, const T D)
    {
      return shadowProjection(light.x, light.y, light.z, A, B, C, D);
    }

    /**
       add m to this
    */
    Matrix<T>& operator+=(const Matrix<T>& m)
    {
      identitymatrix_=false;

      for(int i=0; i<16; i++)
	a[i]+=m[i];

      modified_=true;
      return *this;
    }

    /**
       subtract m from this
    */
    Matrix<T>& operator-=(const Matrix<T>& m)
    {
      identitymatrix_=false;

      for(int i=0; i<16; i++)
	a[i]-=m[i];

      modified_=true;
      return *this;
    }

    /**
       multiply this by m
    */
    Matrix<T>& operator*=(const Matrix<T>& m)
    {
      if(m.identitymatrix_)
	return *this;
      if(identitymatrix_)
	return *this=m;

      // ripped from Mesa3D
      Matrix<T> n;
#define A(row,col)  a[col*4+row]
#define B(row,col)  m.a[col*4+row]
#define P(row,col)  n.a[col*4+row]

#if 0
      if(A(3,0)==0.0 && A(3,1)==0.0 && A(3,2)==0.0 && A(3,3)==1.0
	 && B(3,0)==0.0 && B(3,1)==0.0 && B(3,2)==0.0 && B(3,3)==1.0)
	{
	  for(int i=0; i<3; i++)
	    {
	      const T ai0=A(i,0),  ai1=A(i,1),  ai2=A(i,2),  ai3=A(i,3);
	      P(i,0) = ai0 * B(0,0) + ai1 * B(1,0) + ai2 * B(2,0);
	      P(i,1) = ai0 * B(0,1) + ai1 * B(1,1) + ai2 * B(2,1);
	      P(i,2) = ai0 * B(0,2) + ai1 * B(1,2) + ai2 * B(2,2);
	      P(i,3) = ai0 * B(0,3) + ai1 * B(1,3) + ai2 * B(2,3) + ai3;
	    }
	  P(3,0) = 0;
	  P(3,1) = 0;
	  P(3,2) = 0;
	  P(3,3) = 1;
	}
      else
#endif
	for(int i=0; i<4; i++)
	  {
	    const T ai0=A(i,0),  ai1=A(i,1),  ai2=A(i,2),  ai3=A(i,3);
	    P(i,0) = ai0 * B(0,0) + ai1 * B(1,0) + ai2 * B(2,0) + ai3 * B(3,0);
	    P(i,1) = ai0 * B(0,1) + ai1 * B(1,1) + ai2 * B(2,1) + ai3 * B(3,1);
	    P(i,2) = ai0 * B(0,2) + ai1 * B(1,2) + ai2 * B(2,2) + ai3 * B(3,2);
	    P(i,3) = ai0 * B(0,3) + ai1 * B(1,3) + ai2 * B(2,3) + ai3 * B(3,3);
	  }

#undef A
#undef B
#undef P

      return *this=n;
    }

    /**
       multiply this by f
    */
    Matrix<T>& operator*=(const T& f)
    {
      identitymatrix_=false;

      for(int i=0; i<16; i++)
	a[i]*=f;

      modified_=true;
      return *this;
    }

    /**
       access Matrix element i. The Matrix is organized on row mode.
    */
    T& operator[](const unsigned int i)
    {
      identitymatrix_=false;
      modified_=true;
#ifdef DOJDEBUG
      if(i>15)
	{
	  std::cerr << *this << ".[" << i << "] is wrong" << std::endl;
	  return a[0];
	}
#endif
      return a[i];
    }

    /**
       access Matrix element i. The Matrix is organized on row mode.
    */
    const T& operator[](const unsigned int i) const
    {
#ifdef DOJDEBUG
      if(i>15)
	{
	  std::cerr << *this << ".[" << i << "] is wrong" << std::endl;
	  return a[0];
	}
#endif
      return a[i];
    }

    /**
       access Matrix element (x,y).
    */
    T& operator()(const unsigned int x, const unsigned int y)
    {
      identitymatrix_=false;
      modified_=true;
      return a[y*4+x];
    }

    /**
       access Matrix element (x,y).
    */
    const T& operator()(const unsigned int x, const unsigned int y) const
    {
      return a[y*4+x];
    }

    bool operator==(const Matrix<T>& m) const
    {
      if(this==&m)
	return true;

      if(identitymatrix_ && m.identitymatrix_)
	return true;

      for(int i=0; i<16; i++)
	if(a[i]!=m.a[i])
	  return false;

      return true;
    }

    bool compare(const Matrix<T>& m, const T& eps=0.001) const
    {
      if(this==&m)
	return true;

      if(identitymatrix_ && m.identitymatrix_)
	return true;

      for(int i=0; i<16; i++)
	{
	  const T t=a[i]-m.a[i];
	  if(t<0.0 && t<-eps)
	    return false;
	  if(t>eps)
	    return false;
	}

      return true;
    }

    /**
       write this to ostream os
    */
    std::ostream& dump(std::ostream& os) const
    {
      for(int y=0; y<4; y++)
	{
	  os << ((y==0||y==3)?'(':'|') << " ";

	  for(int x=0; x<4; x++)
	    os << operator()(y, x) << " ";

	  os << ((y==0||y==3)?')':'|');
	  os << std::endl;
	}
      return os;
    }

    /**
       read this from istream is
    */
    std::istream& scan(std::istream& is)
    {
      modified_=true;
      for(int y=0; y<4; y++)
	{
	  const char* starttag[] = { "(", "|", "|", "(" };
	  const char* endtag[] =   { ")", "|", "|", ")" };
	  std::string s;
	  is >> s;
	  if(s!=starttag[y])
	    {
	      std::cerr << "Matrix<T>::scan: '" << starttag[y] << "' expected. found '" << s << "'. aborting..." << std::endl;
	      return is;
	    }

	  for(int x=0; x<4; x++)
	    is >> operator()(y,x);

	  is >> s;
	  if(s!=endtag[y])
	    {
	      std::cerr << "Matrix<T>::scan: '" << endtag[y] << "' expected. found '" << s << "'. aborting..." << std::endl;
	      return is;
	    }
	}

      Matrix<T> n; n.identity();
      identitymatrix_=false; // we need this for the next compare!
      identitymatrix_=(*this==n);
      repaircount=0;

      return is;
    }

  };

  template <typename T>
  inline Matrix<T> operator+(const Matrix<T>& l, const Matrix<T>& r)
  {
    return Matrix<T>(l)+=r;
  }

  template <typename T>
  inline Matrix<T> operator*(const Matrix<T>& a, const Matrix<T>& b)
  {
    if(a.identitymatrix())
      return b;

    if(b.identitymatrix())
      return a;

    return Matrix<T>(a)*=b;
  }

  template <typename T>
  inline Matrix<T> operator*(const Matrix<T>& a, const T b)
  {
    return Matrix<T>(a)*=b;
  }

  template <typename T>
  inline Matrix<T> operator*(const T b, const Matrix<T>& a)
  {
    return Matrix<T>(a)*=b;
  }

  template <typename T>
  inline Matrix<T> operator/(const Matrix<T>& a, const T b)
  {
    return Matrix<T>(a)/=b;
  }

  template <typename T>
  inline std::ostream& operator<<(std::ostream& os, const Matrix<T>& m)
  {
    return m.dump(os);
  }

  template <typename T>
  inline std::istream& operator>>(std::istream& is, Matrix<T>& m)
  {
    return m.scan(is);
  }

  /// convinience typedef for Matrix of floats
  typedef Matrix<float> fmatrix;
  /// convinience typedef for Matrix of doubles
  typedef Matrix<double> dmatrix;

  /**
     this function could possibly be wrong. needs to be checked!
  */
  template <typename T>
  inline Vektor<T> operator*(const Matrix<T>& m, const Vektor<T>& v)
  {
    if(m.identitymatrix())
      return v;

    Vektor<T> n;
    n.x=v.x*m[ 0] + v.y*m[ 4] + v.z*m[ 8] + v.w*m[12];
    n.y=v.x*m[ 1] + v.y*m[ 5] + v.z*m[ 9] + v.w*m[13];
    n.z=v.x*m[ 2] + v.y*m[ 6] + v.z*m[10] + v.w*m[14];
    n.w=v.x*m[ 3] + v.y*m[ 7] + v.z*m[11] + v.w*m[15];
    if (n.w!=0.0 && n.w!=1.0)
      {
	const T a=1.0/n.w;
	n.x *= a;
	n.y *= a;
	n.z *= a;
	n.w = 1;
      }
    return n;
  }

}

#endif
