/* Copyright (c) 2004,2005 Dirk Jagdmann <doj@cubic.org>

This software is provided 'as-is', without any express or implied
warranty. In no event will the authors be held liable for any damages
arising from the use of this software.

Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute it
freely, subject to the following restrictions:

    1. The origin of this software must not be misrepresented; you
       must not claim that you wrote the original software. If you use
       this software in a product, an acknowledgment in the product
       documentation would be appreciated but is not required.

    2. Altered source versions must be plainly marked as such, and
       must not be misrepresented as being the original software.

    3. This notice may not be removed or altered from any source
       distribution. */

// $Header: /code/doj/vektor.hpp,v 1.9 2006/05/12 13:00:56 doj Exp $

#ifndef VEKTOR__HPP
#define VEKTOR__HPP

#include <iostream>
#include <string>

#include <cmath>

#include <constant.hpp>

namespace doj
{
  /**
     Vektor provides a 4 element vektor, which can be used for 3D
     calculations. It is intended for use with Matrix and OpenGL.
     This class might not behave correctly with all calculations when
     the behaviour of the template parameter T differs too much from
     normal number semantics.
  */
  template<typename T>
  class Vektor
  {
  public:
    T x, y, z, w;

    /** default constructor */
    Vektor<T>(const T xx=0, const T yy=0, const T zz=0, const T ww=0) :
      x(xx), y(yy), z(zz), w(ww)
    {
    }

    /** Vektor from (x1,y1,z1) to (x2,y2,z2) */
    Vektor<T>(const T& x1, const T& y1, const T& z1, const T& x2, const T& y2, const T& z2) :
      x(x2-x1), y(y2-y1), z(z2-z1), w(0)
    {
    }

    /** Vektor from a to b */
    Vektor<T>(const Vektor<T>& a, const Vektor<T>& b) :
      x(b.x-a.x), y(b.y-a.y), z(b.z-a.z), w(0)
    {
    }

    /** construct Vektor from first 4 elements of a */
    explicit Vektor<T>(const T *a) :
      x(a[0]), y(a[1]), z(a[2]), w(a[3])
    {
    }

    /** Vektor from a to b */
    Vektor<T>(const T *a, const T *b) :
      x(b[0]-a[0]), y(b[1]-a[1]), z(b[2]-a[2]), w(0)
    {
    }

    /** set Vektor to (xx,yy,zz,ww) */
    Vektor<T>& Init(const T& xx=0, const T& yy=0, const T& zz=0, const T& ww=0)
    {
      x=xx;
      y=yy;
      z=zz;
      w=ww;
      return *this;
    }

    /** set Vektor to (a[0],a[1],a[2],a[3]) */
    Vektor<T>& operator=(const T *a)
    {
      x=a[0];
      y=a[1];
      z=a[2];
      w=a[3];
      return *this;
    }

    Vektor<T>& operator=(const Vektor<T>& v)
    {
      x=v.x;
      y=v.y;
      z=v.z;
      w=v.w;
      return *this;
    }

    /** invert Vektor, thus negating x,y,z */
    Vektor<T>& invert()
    {
      x=-x;
      y=-y;
      z=-z;
      return *this;
    }

    /** sphere coordinates with radius 1
	@param phi horizontal coordinate with range [0..1]
	@param phi vertical coordinate with range [0..1]
    */
    Vektor<T>& sphere(const T& phi, const T& theta)
    {
      const T p=phi*M_PI*2.0;
      const T t=theta*M_PI - M_PI_2;
      x = std::cos(t) * std::cos(p);
      z = std::cos(t) * std::sin(p);
      y = -std::sin(t);
      w=1;
      return *this;
    }

    /** sphere coordinates
	@param phi horizontal coordinate with range [0..r]
	@param phi vertical coordinate with range [0..r]
	@param r radius
    */
    Vektor<T>& sphere(const T& phi, const T& theta, const T& r)
    {
      return sphere(phi, theta) *= r;
    }

    /**
       set this to the cartesian coordinates of v's values interpreted
       as sphere coordinates.
       x and y are mapped from [0..1]. z=0, w=1.

       @param v Vektor containing sphere coordinates. has to be a normalized Vektor!
       @param mult multiplication factor for x and y values
    */
    Vektor<T>& sphere2karth(const Vektor<T>& v, T mult=1)
    {
      mult*=.5;
      x=(M_PI-std::atan2(v.z, v.x)) * M_1_PI * mult;
      y=(-v.y+1.0) * mult;
      z=0;
      w=1;
      return *this;
    }

    /**
       set this to cartesian coordinates of lat and lon with r.
       lon is the angle in the x-z plane. lat gives height on the y axis.

       @param lon in rads
       @param lat in rads
       @param r radius of sphere
    */
    Vektor<T>& sphere2karth(const T& lon, const T& lat, const T& r=1)
    {
      x=std::cos(lon) * std::cos(lat);
      z=std::sin(lon) * std::cos(lat);
      y=std::sin(lat);
      w=1;
      return *this*=r;
    }

    /**
       set this to cartesian coordinates from cylinder coordinates.
       Not implemented yet!
     */
    Vektor<T>& cylinder(const T& phi, const T& h, const T& r1, const T& r2)
    {
      std::cerr << "Vektor<T>::cylinder(): not implemented yet!!!" << std::endl;
      y=h;
      w=1;
      return *this;
    }

    /**
       construct cartesian coordinates from torus coordinates
       @param phi horizontal torus coordinate
       @param theta vertical torus coordinate
       @param r0 radius of inner ring
       @param r1 radius of outer ring
    */
    Vektor<T>& torus(const T& phi, const T& theta, const T& r0, const T& r1)
    {
      const T p=phi*M_PI*2.0;
      const T t=theta*M_PI*2.0;
      x = std::cos(theta) * ( r0 + r1 * std::cos(phi) );
      y = std::sin(theta) * ( r0 + r1 * std::cos(phi) );
      z = std::sin(phi) * r1;
      w=1;
      return *this;
    }

    /**
       For reference see <a href="http://astronomy.swin.edu.au/pbourke/modelling/torus/">http://astronomy.swin.edu.au/pbourke/modelling/torus/</a>
       @param r0 radius of inner ring
       @param r1 radius of outer ring
    */
    Vektor<T>& supertorus(const T& phi, const T& theta, const T& r0, const T& r1, const T& n1, const T& n2)
    {
      const T p=phi*M_PI*2.0;
      const T t=theta*M_PI*2.0;
      x = std::pow(std::cos(theta), n1) * ( r0 + r1 * std::pow(std::cos(phi), n2) );
      y = std::pow(std::sin(theta), n1) * ( r0 + r1 * std::pow(std::cos(phi), n2) );
      z = std::pow(std::sin(phi), n2) * r1;
      w=1;
      return *this;
    }

    /**
       For reference see <a href="http://astronomy.swin.edu.au/pbourke/modelling/superellipse/">http://astronomy.swin.edu.au/pbourke/modelling/superellipse/</a>
       @param normal if true, the Vektor becomes the normal at the specific point. if false the Vektor becomes the specific point.
    */
    Vektor<T>& superellipsoid(const T& phi, const T& theta, const T& rx, const T& ry, const T& rz, const T& n1, const T& n2, const bool normal=false)
    {
      const T p=phi*M_PI - M_PI_2;
      const T t=theta * M_PI * 2.0 - M_PI;

      if(normal)
	{
	  x = 1.0/rx * std::pow(std::cos(p), 2.0-n1) * std::pow(std::cos(t), 2.0-n2);
	  y = 1.0/ry * std::pow(std::cos(p), 2.0-n1) * std::pow(std::sin(t), 2.0-n2);
	  z = 1.0/rz * std::pow(std::sin(p), 2.0-n1);
	  w=0;
	}
      else
	{
	  x = rx * std::pow(std::cos(p), n1) * std::pow(std::cos(t), n2);
	  y = ry * std::pow(std::cos(p), n1) * std::pow(std::sin(t), n2);
	  z = rz * std::pow(std::sin(p), n1);
	  w=1;
	}

      return *this;
    }

    /**
       For reference see <a href="http://astronomy.swin.edu.au/pbourke/modelling/teardrop/">http://astronomy.swin.edu.au/pbourke/modelling/teardrop/</a>
    */
    Vektor<T>& teardrop(const T& phi, const T& theta)
    {
      const T p=phi*M_PI*2.0;
      const T t=theta*M_PI;
      x=.5 * (1-std::cos(t)) * std::sin(t) * std::cos(p);
      y=.5 * (1-std::cos(t)) * std::sin(t) * std::sin(p);
      z=std::cos(t);
      w=1;
      return *this;
    }

    /**
       For reference see <a href="http://astronomy.swin.edu.au/pbourke/modelling/spring/">http://astronomy.swin.edu.au/pbourke/modelling/spring/</a>
    */
    Vektor<T>& spring(const T& phi, const T& theta, const T& r1, const T& r2, const T& length)
    {
      const T p=phi*M_PI*2.0;
      const T t=theta*M_PI*2.0;
      x = (1.0 - r1 * std::cos(theta)) * std::cos(phi);
      y = (1.0 - r1 * std::cos(theta)) * std::sin(phi);
      z = r2 * (std::sin(theta) + length * phi * M_1_PI);
      w=1;
      return *this;
    }

#if 0 // todo: there's an error in the formulas
    /**
       For reference see <a href="http://astronomy.swin.edu.au/pbourke/modelling/shell/">http://astronomy.swin.edu.au/pbourke/modelling/shell/</a>
       @param n number of rounds
       @param rf final radius
       @param ri inner radius
       @param height height
    */
    Vektor<T>& spiral(const T& phi, const T& theta, const T& n, const T& rf, const T& ri, const T& height)
    {
      const T p=phi*M_PI*2.0;
      const T t=theta*M_PI*2.0;
      x=rf*(1.0-t*M_1_PI*.5) * std::cos(n*t) * (1+std::cos(s)) + ri*std::cos(n*t);
      y=rf*(1.0-t*M_1_PI*.5) * std::sin(n*t) * (1+std::cos(s)) + ri*std::sin(n*t);
      z=height*(t*M_1_PI*.5) + rf*(1.0-t*M_1_PI*.5) * std::sin(s);
      w=1;
      return *this;
    }
#endif

    /**
       For reference see <a href="http://astronomy.swin.edu.au/pbourke/curves/cycloid/">http://astronomy.swin.edu.au/pbourke/curves/cycloid/</a>
       Note: z is always 0
    */
    Vektor<T>& cycloid(const T& theta, const T& r)
    {
      const T t=theta*M_PI*2.0;
      x=r*(t-std::sin(t));
      y=r*(1.0-std::cos(t));
      z=0;
      w=1;
      return *this;
    }

    /// clamp this into range [min, max]
    Vektor<T>& clamp(const T& minval=0, const T& maxval=1)
    {
      x=min(max(x, minval), maxval);
      y=min(max(y, minval), maxval);
      z=min(max(z, minval), maxval);
      w=min(max(w, minval), maxval);
      return *this;
    }

    /**
       add a to this
     */
    Vektor<T>& operator+=(const Vektor<T>& a)
    {
      x+=a.x;
      y+=a.y;
      z+=a.z;
      return *this;
    }

    /**
       add (a[0],a[1],a[2]) to this
    */
    Vektor<T>& operator+=(const T *a)
    {
      x+=a[0];
      y+=a[1];
      z+=a[2];
      return *this;
    }

    /** @return negation of this */
    Vektor<T> operator-() const
    {
      return Vektor<T>(-x, -y, -z, -w);
    }

    /**
       subtract a from this
    */
    Vektor<T>& operator-=(const Vektor<T>& a)
    {
      x-=a.x;
      y-=a.y;
      z-=a.z;
      w=0;
      return *this;
    }

    /**
       subtract (a[0],a[1],a[2]) from this
    */
    Vektor<T>& operator-=(const T *a)
    {
      x-=a[0];
      y-=a[1];
      z-=a[2];
      w=0;
      return *this;
    }

    /**
       multiply this by a
    */
    Vektor<T>& operator*=(const T& a)
    {
      x*=a;
      y*=a;
      z*=a;
      return *this;
    }

    /**
       divide this by a
    */
    Vektor<T>& operator/=(const T& a)
    {
      x/=a;
      y/=a;
      z/=a;
      return *this;
    }

    /** @return length of this */
    double length() const
    {
      return std::sqrt(x*x+y*y+z*z);
    }

    /** @return x*x+y*y+z*z, which is a fast approximation of the length */
    double lengthFast() const
    {
      return x*x+y*y+z*z;
    }

    /**
       change then length of this to len.
    */
    Vektor<T>& normalise(const T& len=1)
    {
      T l=length();
      if(l!=static_cast<T>(0.0))
	{
	  l=len/l;
	  x*=l;
	  y*=l;
	  z*=l;
	}
      return *this;
    }

    /**
       normalise this. The length of this is 1 after this operation.
    */
    Vektor<T>& unit()
    {
      return normalise();
    }

    /**
       @return the dot product of this and a
     */
    T dot(const Vektor<T>& a) const
    {
      return x*a.x+y*a.y+z*a.z;
    }

    /**
       @return the cross product of this and a
    */
    Vektor<T> cross(const Vektor<T>& a) const
    {
      return Vektor<T>(y*a.z - z*a.y, z*a.x - x*a.z, x*a.y - y*a.x);
    }

    /**
       @return the angle between this and a
    */
    T angle(const Vektor<T>& a) const
    {
      //return acos(dot(a)/(length()*a.length()));
      return std::acos( dot(a) / std::sqrt((x*x+y*y+z*z) * (a.x*a.x+a.y*a.y+a.z*a.z)) );
    }

    /**
       @return the angle between this and x axis
     */
    T angleX() const
    {
      return std::acos(x/length());
    }

    /**
       @return the angle between this and y axis
     */
    T angleY() const
    {
      return std::acos(y/length());
    }

    /**
       @return the angle between this and z axis
     */
    T angleZ() const
    {
      return std::acos(z/length());
    }

    /** write this to ostream os */
    virtual std::ostream& dump(std::ostream& os) const
    {
      return os << "<Vektor> " << x << " " << y << " " << z << " " << w << " </Vektor>";
    }

    /** read this from ostream os */
    virtual std::istream& scan(std::istream& is)
    {
      std::string s;
      is >> s;
      if(s!="<Vektor>")
	{
	  std::cerr << "Vektor::scan(): did not find start tag 'Vektor::('. aborting..." << std::endl;
	  return is;
	}

      is >> x >> y >> z >> w;

      is >> s;
      if(s!="</Vektor>")
	std::cerr << "Vektor::scan(): did not find end tag ')'. aborting..." << std::endl;

      return is;
    }

    /** @return whether this and a are equal */
    const bool operator==(const Vektor<T>& a) const
    {
      return x==a.x && y==a.y && z==a.z && w==a.w;
    }


    /**
       This method is inefficient! Use direct access of x,y,z and w if possible
       @return the internal instance variable x, y, z or w depending on i.
    */
    T& operator[](const int i)
    {
      switch(i)
	{
	case 0: return x;
	case 1: return y;
	case 2: return z;
	default:
	case 3: return w;
	}
    }

  };

  template<typename T>
  inline std::ostream& operator<<(std::ostream& os, const Vektor<T>& v)
  {
    return v.dump(os);
  }

  template<typename T>
  inline std::istream& operator>>(std::istream& is, Vektor<T>& v)
  {
    return v.scan(is);
  }

  template<typename T>
  inline Vektor<T> operator+(const Vektor<T>& a, const Vektor<T>& b)
  {
    return Vektor<T>(a)+=b;
  }

  template<typename T>
  inline Vektor<T> operator-(const Vektor<T>& a, const Vektor<T>& b)
  {
    return Vektor<T>(a)-=b;
  }

  template<typename T>
  inline Vektor<T> operator*(const Vektor<T>& a, const T b)
  {
    return Vektor<T>(a)*=b;
  }

  template<typename T>
  inline Vektor<T> operator*(const T b, const Vektor<T>& a)
  {
    return Vektor<T>(a)*=b;
  }

  template<typename T>
  inline Vektor<T> operator/(const Vektor<T>& a, const T b)
  {
    return Vektor<T>(a)/=b;
  }

  /// convienience typedef for Vektor of floats
  typedef Vektor<float> fvector;
  /// convienience typedef for Vektor of floats
  typedef Vektor<double> dvector;

}

#endif
