#include <iostream>
using namespace std;

#include "font.h"
#include "tga.h"

//#include "doj/math.h"
using namespace doj;

FT_Library Font::library;
bool Font::libraryInit=false;
FT_Face Font::face;

const int Font::Left=0;
const int Font::Center=1;
const int Font::Right=2;

const int FontQuality=128;

Font::Font(const string& n) : 
  ColorObject(n),
  shaded(false)
{
  textureEnable=true;
  blendEnable=true;
  for(int i=0; i<128; i++)
    {
      charAdvance[i]=charWidth[i]=charHeight[i]=ty[i]=0;
      textures[i]=0;
    }
}

void Font::deinit()
{
  glDeleteTextures(128, textures);
  Object::deinit();
}

int Font::init()
{
  cerr << "Font::init(): you should not use this function" << endl;
  return -1;
}

int Font::init(const FT_Byte *font, const int fontLen, const int fontindex)
{
  int error;

  if(!libraryInit)
    {
      error = FT_Init_FreeType( &library );
      if ( error )
	{
	  cerr << "Font::init(): could not init Fretype2" << endl;
	  return -1;
	}
      libraryInit=true;
    }

  error = FT_New_Memory_Face(library, font, fontLen, fontindex, &face);
  if ( error == FT_Err_Unknown_File_Format )
    {
      cerr << "Font::init(): could not open font unknown format" << endl;
      return -1;
    }
  else if ( error )
    {
      cerr << "Font::init(): error while opening font" << endl;
      return -1;
    }

  error = FT_Set_Pixel_Sizes( face, 0, FontQuality );
  if(error)
    {
      cerr << "Font::init(): could not set pixel size" << endl;
      return -1;
    }

  const float scale=1.0/(64.0*FontQuality);
  for(int i=32; i<128; i++)
    {
      error = FT_Load_Char( face, i, FT_LOAD_RENDER );
      if(error)
	{
	  cerr << "Font::init(): could not render glyph " << i << " " << (char)i << endl;
	  continue;
	}

      charAdvance[i]=face->glyph->metrics.horiAdvance * scale;
      charWidth[i]  =face->glyph->metrics.width * scale;
      charHeight[i] =face->glyph->metrics.height * scale;
      ty[i]         =face->glyph->metrics.horiBearingY * scale;
      //face->glyph->metrics.horiBearingX * scale;

      const int w=face->glyph->bitmap.width;
      const int h=face->glyph->bitmap.rows;
      if(w>0 && h>0)
	textures[i]=makeTexture(face->glyph->bitmap.buffer, w, h, GL_ALPHA);

    }

  charAdvance[static_cast<int>('\t')]=8*charAdvance[static_cast<int>(' ')];

  return 0;
}

GLfloat Font::width(const char c) const
{
  return (c>=0)?charAdvance[static_cast<int>(c)]:0;
}

GLfloat Font::width(const string& str) const
{
  GLfloat w=0;
  for(string::const_iterator i=str.begin(); i!=str.end(); ++i)
    w+=width(*i);
  return w;
}

GLfloat Font::print(const char c, const GLfloat x, const GLfloat y, const GLfloat z, const int align) const
{
  const GLfloat w=width(c);
  if(w>0)
    {
      glPushMatrix();
      float xx=x;
      switch(align)
	{
	case Left: break;
	case Center: xx-=w/2.0; break;
	case Right: xx-=w; break;
	default: cerr << "Font::print(char,...): wrong align argument" << endl;
	}
      glTranslatef(xx, y, z);
      draw(c);
      glPopMatrix();
    }
  return w;
}

GLfloat Font::print(const string& str, const GLfloat x, const GLfloat y, const GLfloat z, const int align) const
{
  if(str.empty())
    return 0;

  const GLfloat w=width(str);
  if(w==0)
    return 0;

  glPushMatrix();
  glTranslatef(x, y, z);

  switch(align)
    {
    case Left: glTranslatef(w, 0, 0); break;
    case Center: glTranslatef(w/2.0, 0, 0); break;
    case Right: break;
    default: cerr << "Font::print(): invalid font align parameter: " << align << endl;
    }

  for(string::const_reverse_iterator i=str.rbegin(); i!=str.rend(); ++i)
    {
      const GLfloat w=width(*i);
      if(w>0)
	{
	  glTranslatef(-w, 0, 0);
	  draw(*i);
	}
    }

  glPopMatrix();

  return w;
}

void Font::draw(const char c) const
{
  if(c<32)
    return;

  glBindTexture(GL_TEXTURE_2D, textures[static_cast<int>(c)]);
  glBegin(GL_TRIANGLES);
  {
    const float e1=0.01;
    const float e2=0.99;

    setMaterial();

    // first
    glColor4f(r,g,b,a);
    glTexCoord2f(e1, e2); glVertex2f(0, ty[static_cast<int>(c)]-charHeight[static_cast<int>(c)]); // 1

    if(shaded)
      glColor4f(colorDark.r, colorDark.g, colorDark.b, colorDark.a);
    glTexCoord2f(e2, e2); glVertex2f(charWidth[static_cast<int>(c)], ty[static_cast<int>(c)]-charHeight[static_cast<int>(c)]); // 2

    if(shaded)
      glColor4f(r,g,b,a);
    glTexCoord2f(e2, e1); glVertex2f(charWidth[static_cast<int>(c)], ty[static_cast<int>(c)]); // 3

    // second
    glTexCoord2f(e2, e1); glVertex2f(charWidth[static_cast<int>(c)], ty[static_cast<int>(c)]); // 3

    if(shaded)
      glColor4f(colorLight.r, colorLight.g, colorLight.b, colorLight.a);
    glTexCoord2f(e1, e1); glVertex2f(0, ty[static_cast<int>(c)]); // 4

    if(shaded)
      glColor4f(r,g,b,a);
    glTexCoord2f(e1, e2); glVertex2f(0, ty[static_cast<int>(c)]-charHeight[static_cast<int>(c)]); // 1
  }
  glEnd();
}
