#if MUSIC
#include <fmod.h>
#include <fmod_errors.h>
#ifdef __linux__
#include <wincompat.h>
#endif
#endif

#include <cstdio>

#ifdef __linux__
#include <unistd.h>
#endif

#include "anfang.h"
#include "camera.h"
#include "flash.h"
#include "font.h"
#include "grid.h"
#include "last.h"
#include "main.h"
#include "mouse.h"
#include "rotor.h"
#include "space2.h"
#include "starfield.h"
#include "swoosh.h"
#include "tga.h"
#include "time.h"

#include "doj/gl.hpp"
using namespace doj;

#include "data.h"		// must be included at last

/// don't display any graphics, only record music sync events into the music.rec file
#define DOJRECORD 0

#if DOJRECORD && !MUSIC
#error You cannot record events without music activated
#endif

Screen screen = { 640, 480 };

static int winmain=0;

Font *font1=0;
Font *font2=0;
static Grid *grid=0;

static Flash *flash=0;
static float esctime=2;
static int esc=-1;

static bool gridEnable=false;
static bool split=false;

static Mouse mouseLeft;
static Mouse mouseMiddle;
static Mouse mouseRight;

static Camera camera, cameraBak;
Camera *activeCamera=&camera;

static int currentEffect=0;

Object *blendObject=0;

int TextureMemorySize=32;

bool lowtech=false;
bool stereo=false;

static bool showframes=false;

#if MUSIC
static FMUSIC_MODULE *mod=0;
static int startorder=0;
#else
typedef void FMUSIC_MODULE;
#endif

#if DOJRECORD
FILE *record=0;
int recordtime=0;
#endif

modSync_t modSync;

rgba clearColor(0,0,0);

void escPressed()
{
  esc=timeGetTime();
}

void modCallback(FMUSIC_MODULE *mod, unsigned char param)
{
  modSync.push_back(param);
#if DOJRECORD
  fprintf(record, "{%i,%i},\n", recordtime, param);
#endif
}

void scene(const int time)
{
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();

  switch(currentEffect)
    {
    default:
    case 1:
      Space2::draw(time);
      break;
    case 4:
      Space2::draw(time);
      Anfang::draw(time);
      break;
    case 2: Anfang::draw(time); break;
    case 3: Swoosh::draw(time); break;
    case 5: Starfield::draw(time); break;
    case 6: Rotor::draw(time); break;
    case 0xf: Last::draw(time); break;
    }

  if(grid && gridEnable)
    {
      grid->before();
      grid->draw();
      grid->after();
    }
}

static void init();

#if !MUSIC
struct syncevent {
  int time;
  unsigned char param;
};
#endif

void routine()
{
  static bool initialized=false;
  if(!initialized)
    {
      init();
      initialized=true;
    }

  const int time=timeGetTime();
  static int starttime=-1;
  if(starttime==-1)
    starttime=time;

#if DOJRECORD
  recordtime=time-starttime;
#endif

  if(showframes)
    {
  static int oldtime=0;
  const int frames=static_cast<int>(1.0/((time-oldtime)/1000.0));
  static int oldframes=0;
  oldframes=(oldframes+frames)/2;
#if _WIN32
  static int z=0;
  if(++z==10)
    {
      z=0;
#endif
      clog << "\r";
#if MUSIC
      clog << FMUSIC_GetOrder(mod) << ":" << FMUSIC_GetRow(mod) << " ";
#endif
      clog << oldframes << " frames ";
#if MUSIC && defined(_WIN32)
      clog << FSOUND_GetCPUUsage() << "% FMOD ";
#endif
      clog << "    " << flush;
#if _WIN32
    }
#endif
  oldtime=time;
    }

  // If we have no music, we have to fake the syncs
#if !MUSIC
  static const struct syncevent synceventlist[] = {
#include "music.rec"
  };
  static int synceventindex=0;
  const int t=time-starttime;
  while(t>=synceventlist[synceventindex].time)
    modCallback(NULL, synceventlist[synceventindex++].param);
#endif

  // don't make any OpenGL output, if we want to record the music syncs
#if !DOJRECORD
  for(modSync_t::iterator i=modSync.begin(); i!=modSync.end(); ++i)
    {
      const int e=*i>>4;
      if(e>0)
	currentEffect=e;
    }

  glClearColor(clearColor.r, clearColor.g, clearColor.b, clearColor.a);
  glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);

  if(split)
    {
      //Perspective
      glViewport(screen.width/2, 0, screen.width/2, screen.height/2);
      scene(time);

      // Front
      Camera ct=*activeCamera;
      ct.head=ct.roll=ct.pitch=0;
      glViewport(screen.width/2, screen.height/2, screen.width/2, screen.height/2);
      scene(time);

      // Top
      ct=*activeCamera;
      ct.head=-90;
      ct.roll=ct.pitch=0;
      glViewport(0, screen.height/2, screen.width/2, screen.height/2);
      scene(time);

      // Left
      ct=*activeCamera;
      ct.pitch=90;
      ct.roll=ct.head=0;
      glViewport(0, 0, screen.width/2, screen.height/2);
      scene(time);
    }
  else
    {
      glViewport(0, 0, screen.width, screen.height);
      if(stereo)
	{
	  // left eye
	  Camera::left=true; Camera::right=false;
	  glColorMask(GL_TRUE, GL_FALSE, GL_FALSE, GL_TRUE);
	  scene(time);

	  glClear(GL_DEPTH_BUFFER_BIT);

	  // right eye
	  Camera::left=false; Camera::right=true;
	  glColorMask(GL_FALSE, GL_TRUE, GL_TRUE, GL_TRUE);
	  scene(time);

	  glColorMask(GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);
	}
      else
	{
	  Camera::left=false; Camera::right=false;
	  scene(time);
	}
    }

  // check for esc pressed
  if(esc>-1)
    {
      const float e=(time-esc)/1000.0/esctime;
      if(e>1)
	exit(0);

      flash->before();
      flash->draw(time);
      flash->after();
#if MUSIC
      FMUSIC_SetMasterVolume(mod, static_cast<int>((1.0-e)*256.0));
#endif
    }

  glutSwapBuffers();

#else // #if !DOJRECORD
  if(esc>-1)
    exit(0);
#endif
}

void visible(int state)
{
  switch(state)
    {
    case GLUT_NOT_VISIBLE:
      glutIdleFunc(0);
      break;
    case GLUT_VISIBLE:
    default:
      glutIdleFunc(routine);
      break;
    }
}

void key(unsigned char key, int x, int y)
{
  switch(key)
    {
    case 27:
    case 'q': escPressed(); break;
#if DOJDEBUG
    case 'f': glutFullScreen(); break;
    case 'w':
      glutReshapeWindow(screen.width, screen.height);
      glutPositionWindow((1024-screen.width)/2, 32);
      break;
    case 's': stereo=!stereo; break;
    case 'g': gridEnable=!gridEnable; break;
    case 'r': activeCamera->reset(); cameraBak.reset(); break;
    case 'c': cout << endl << *activeCamera; break;
    case '\t': split=!split; break;
#if MUSIC
    case 'p':
      {
	static bool pause=false;
	FMUSIC_SetPaused(mod,(pause=!pause));
      }
      break;
#endif
    case 'a':
      if(blendObject)
	{
	  GLenum src[] =
	    {
	      GL_ONE,
	      GL_SRC_COLOR,
	      GL_SRC_ALPHA,
	      GL_ONE_MINUS_SRC_ALPHA,
	    };
	  char* srcs[] =
	    {
	      "GL_ONE",
	      "GL_SRC_COLOR",
	      "GL_SRC_ALPHA",
	      "GL_ONE_MINUS_SRC_ALPHA",
	    };
	  GLenum dst[] =
	    {
	      GL_ZERO,
	      GL_ONE,
	      GL_DST_COLOR,
	      GL_ONE_MINUS_DST_COLOR,
	      GL_DST_ALPHA,
	      GL_ONE_MINUS_DST_ALPHA,
	    };
	  char* dsts[] =
	    {
	      "GL_ZERO",
	      "GL_ONE",
	      "GL_DST_COLOR",
	      "GL_ONE_MINUS_DST_COLOR",
	      "GL_DST_ALPHA",
	      "GL_ONE_MINUS_DST_ALPHA",
	    };
	  static int s=0, d=0;

	  blendObject->blendSrc=src[s];
	  blendObject->blendDst=dst[d];
	  clog << endl << srcs[s] << "\t" << dsts[d] << endl;
	  (++d)%=sizeof(dst)/sizeof(GLenum);
	  if(d==0)
	    (++s)%=sizeof(src)/sizeof(GLenum);
	}
      break;
#endif
    }
#if DOJDEBUG
  if(key>='0' && key<='9')
    activeCamera->fov=(key-'0')*19.9;
#endif
}

void SpecialFunc(int key, int x, int y)
{
  switch(key)
    {
#if MUSIC
    case GLUT_KEY_LEFT:
      {
	int i=FMUSIC_GetOrder(mod);
	if(i>0)
	  FMUSIC_SetOrder(mod, i-1);
      }
      break;
    case GLUT_KEY_RIGHT:
      FMUSIC_SetOrder(mod, FMUSIC_GetOrder(mod)+1);
      break;
#endif
#if 0
    case GLUT_KEY_UP:
      break;
    case GLUT_KEY_DOWN:
      break;
    case GLUT_KEY_PAGE_UP:
      break;
    case GLUT_KEY_PAGE_DOWN:
      break;
    case GLUT_KEY_HOME:
      break;
    case GLUT_KEY_END:
      break;
    case GLUT_KEY_INSERT:
      break;
    case GLUT_KEY_F1:
    case GLUT_KEY_F2:
    case GLUT_KEY_F3:
    case GLUT_KEY_F4:
    case GLUT_KEY_F5:
    case GLUT_KEY_F6:
    case GLUT_KEY_F7:
    case GLUT_KEY_F8:
    case GLUT_KEY_F9:
    case GLUT_KEY_F10:
    case GLUT_KEY_F11:
    case GLUT_KEY_F12:
      break;
#endif
    }
}

void reshape(int width, int height)
{
  screen.width=width;
  screen.height=height;
}

void mouse1(int button, int state, int x, int y)
{
  switch(button)
    {
    case GLUT_LEFT_BUTTON:
      mouseLeft.button=(state==GLUT_DOWN)?true:false;
      mouseLeft.x=x;
      mouseLeft.y=y;
      if(mouseLeft.button)
	cameraBak=*activeCamera;
      break;
    case GLUT_MIDDLE_BUTTON:
      mouseMiddle.button=(state==GLUT_DOWN)?true:false;
      mouseMiddle.x=x;
      mouseMiddle.y=y;
      if(mouseMiddle.button)
	cameraBak=*activeCamera;
      break;
    case GLUT_RIGHT_BUTTON:
      mouseRight.button=(state==GLUT_DOWN)?true:false;
      mouseRight.x=x;
      mouseRight.y=y;
      if(mouseRight.button)
	cameraBak=*activeCamera;
      break;
    }
}

void mouse2(int x, int y)
{
  const float accelMove=10.0;
  const float accelRot=50.0;

  if(mouseLeft.button)
    {
      activeCamera->x=cameraBak.x + (float)(x-mouseLeft.x)/(float)screen.width*accelMove;
      activeCamera->y=cameraBak.y - (float)(y-mouseLeft.y)/(float)screen.height*accelMove;
    }

  if(mouseMiddle.button)
    {
      activeCamera->z=cameraBak.z - (float)(y-mouseMiddle.y)/(float)screen.height*accelMove;
      activeCamera->pitch=cameraBak.pitch - (float)(x-mouseMiddle.x)/(float)screen.width*accelRot;
    }
  if(mouseRight.button)
    {
      activeCamera->roll=cameraBak.roll - (float)(x-mouseRight.x)/(float)screen.width*accelRot;
      activeCamera->head=cameraBak.head - (float)(y-mouseRight.y)/(float)screen.height*accelRot;
    }
}

static void deinit()
{
#if MUSIC
  if(mod)
    {
      FMUSIC_StopSong(mod);
      FMUSIC_FreeSong(mod);
      FSOUND_Close();
    }
#endif

#if DOJRECORD
  if(record)
    fclose(record);
#endif

  if(font1)
    font1->deinit(), delete font1;
  if(font2)
    font2->deinit(), delete font2;

  if(grid)
    grid->deinit(), delete grid;

  if(winmain)
    glutDestroyWindow(winmain);

#ifdef DOJDEBUG
#ifndef _WIN32
  clog << endl;
  clog << "used ";
  clog << TextureSize();
  clog << " = ";
  clog << TextureSize()/1024/1024;
  clog << "MB for textures" << endl;
#endif
#endif
}

#if defined(_WIN32) && (!defined(DOJDEBUG))
int APIENTRY WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nCmdShow)
{
  char *argv[] = { "nix.exe", 0 };
  int argc=1;
#else
  static bool SoundAlsa=true, SoundOss=true, SoundEsd=true;
int main(int argc, char **argv)
{
#endif

  atexit(deinit);

  glutInit(&argc, argv);

  bool fullscreen=false;
#ifdef DOJDEBUG
  showframes=true;
#else
  fullscreen=true;
#endif

  for(int i=1; i<argc; i++)
    {
      string s=argv[i];
      if(s=="-16")
	TextureMemorySize=16;
      else if(s=="-8")
	TextureMemorySize=8;

      else if(s=="-w")
	{
	  fullscreen=false;
	  if(i+1<argc && isdigit(argv[i+1][0]))
	    {
	      i++;
	      screen.width=atoi(argv[i]);
	      int p=0;
	      while(argv[i][p] && argv[i][p]!='x') p++;
	      p++;
	      screen.height=atoi(argv[i]+p);
	    }
	}
      else if(s=="-f")
	fullscreen=true, showframes=false;

      else if(s=="-l")
	lowtech=true;

      else if(s=="-s")
	stereo=true;

      else if(s=="-d")
	showframes=true;

      else if(s=="--alsa")
	SoundAlsa=true, SoundOss=SoundEsd=false;
      else if(s=="--oss")
	SoundOss=true, SoundAlsa=SoundEsd=false;
      else if(s=="--esd")
	SoundEsd=true, SoundAlsa=SoundOss=false;
      else if(s=="--no")
	SoundAlsa=SoundOss=SoundEsd=false;
      else
	{
	  cout << "NiX OpenGL Demo" << endl;
	  cout << "args: -8   8MB texture memory" << endl;
	  cout << "      -16  16MB texture memory" << endl;
	  cout << "      -f   fullscreen mode" << endl;
	  cout << "      -w   window mode" << endl;
	  cout << "      -l   lowtech or light version" << endl;
	  cout << "      -s   stereo rendering" << endl;
	  cout << "      -d   show frames/sec" << endl;
#if !defined(_WIN32)
	  cout << "--alsa,--oss,--esd,--no use sound system" << endl;
#endif
	  return 1;
	}
    }

  glutInitWindowPosition((1024-screen.width)/2, 32);
  glutInitWindowSize(screen.width, screen.height);
  glutInitDisplayMode(GLUT_RGBA|GLUT_ALPHA|GLUT_DEPTH|GLUT_DOUBLE);

  winmain=glutCreateWindow("NiX");

  glutDisplayFunc(routine);
  glutIdleFunc(routine);
  glutReshapeFunc(reshape);
  glutVisibilityFunc(visible);
  glutKeyboardFunc(key);
#if DOJDEBUG
  glutSpecialFunc(SpecialFunc);
  glutMouseFunc(mouse1);
  glutMotionFunc(mouse2);
#endif

  if(fullscreen)
    glutFullScreen();

  glutMainLoop();
  return 0;
}

void e(const char *s)
{
  const GLenum err = glGetError();
  if(err)
    cerr << s << " " << gluErrorString(err) << endl;
}

static void inittick()
{
  static int tick=0;
  tick++;

  string s="initializing";
  for(int i=0; i<tick; i++)
    s+='.';

  glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);
  glViewport(0, 0, screen.width, screen.height);
  gluOrthoBefore(screen.width, screen.height);
  {
    font1->color(1,1,1,1);
    font1->before();
    glTranslatef(0,0,1);
    glScale(.2);
    font1->print("NiX", 0, .5, 0, Font::Center);
    font1->print(s, -3, -.5);
    font1->after();
  }
  gluOrthoAfter();
  glutSwapBuffers();
}

static void init()
{
  glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
  glFrontFace(GL_CCW);

  //glEnable(GL_ALPHA_TEST); // it seems we don't need those two
  //glAlphaFunc(GL_GREATER, 0.5);

  // schaltet specular hightlights auf texturen ein!
  //glLightModeli(GL_LIGHT_MODEL_COLOR_CONTROL, GL_SEPARATE_SPECULAR_COLOR);

  glGetIntegerv(GL_MAX_LIGHTS, &gl_max_lights);

  glMatrixMode(GL_TEXTURE);
  glLoadIdentity();
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();

  font1=new Font("Font1");
  if(font1==0 || font1->init(static_cast<const FT_Byte*>(&TTFariblk), TTFariblkLEN) < 0)
    {
      cerr << "main(): could not init Font1" << endl;
      exit(1);
    }
  font1->color(0,1,0);
  font1->shaded=true;
  inittick();

#if MUSIC
  if (FSOUND_GetVersion() < FMOD_VERSION)
    {
      cerr << "Error : You are using the wrong DLL version!  You should be using FMOD " << FMOD_VERSION << endl;
      exit(1);
    }

#if !defined(_Win32)
  if(SoundAlsa)
    FSOUND_SetOutput(FSOUND_OUTPUT_ALSA);

  else if(SoundOss)
    FSOUND_SetOutput(FSOUND_OUTPUT_OSS);
  else if(SoundEsd)
    FSOUND_SetOutput(FSOUND_OUTPUT_ESD);
  else
    FSOUND_SetOutput(FSOUND_OUTPUT_NOSOUND);
#endif

  if (!FSOUND_Init(44100, 32, FSOUND_INIT_GLOBALFOCUS))
    {
      cerr << FMOD_ErrorString(FSOUND_GetError()) << endl;
      exit(1);
    }
  mod=FMUSIC_LoadSongEx(reinterpret_cast<char*>(&XMrp_dawnv), 0, XMrp_dawnvLEN, FSOUND_LOADMEMORY, 0,0);
  if (!mod)
    {
      cerr << FMOD_ErrorString(FSOUND_GetError()) << endl;
      exit(1);
    }
  if(!FMUSIC_SetZxxCallback(mod, modCallback))
    {
      cerr << FMOD_ErrorString(FSOUND_GetError()) << endl;
      exit(1);
    }
#if DOJDEBUG
  char *s=getenv("STARTORDER");
  if(s)
    startorder=atoi(s);
#endif
  inittick();
#endif // MUSIC

#if DOJRECORD
  record=fopen("music.rec", "w");
  if(record==0)
    {
      cerr << "main(): could not open music.rec" << endl;
      exit(1);
    }
#endif

  cameraBak=*activeCamera;

  grid=new Grid("Grid");
  if(grid==0 || grid->init() < 0)
    {
      cerr << "main(): could not init Grid" << endl;
      exit(1);
    }
  inittick();

  flash=new Flash();
  if(flash==NULL)
    {
      cerr << "main(): could not alloc flash" << endl;
      exit(1);
    }
  flash->from(0,0,0,0);
  flash->to(0,0,0,1);
  flash->time(esctime);
  inittick();

  if(Space2::init() < 0)
    {
      cerr << "main(): could not init Space2" << endl;
      exit(1);
    }
  inittick();

  if(Anfang::init() < 0)
    {
      cerr << "main(): could not init Anfang" << endl;
      exit(1);
    }
  inittick();

  if(Swoosh::init() < 0)
    {
      cerr << "main(): could not init Swoosh" << endl;
      exit(1);
    }
  inittick();

  if(Starfield::init() < 0)
    {
      cerr << "main(): could not init Starfield" << endl;
      exit(1);
    }
  inittick();

  if(Rotor::init() < 0)
    {
      cerr << "main(): could not init Rotor" << endl;
      exit(1);
    }
  inittick();

  if(Last::init() < 0)
    {
      cerr << "main(): could not init Last" << endl;
      exit(1);
    }
  inittick();

#if DOJDEBUG
  clog << endl;
#endif

#if MUSIC
  FMUSIC_PlaySong(mod);
  if(startorder>0)
    if(!FMUSIC_SetOrder(mod, startorder))
      {
	cerr << "could not set order " << startorder << ": " << FMOD_ErrorString(FSOUND_GetError()) << endl;
	exit(1);
      }
#endif
}
