#include <cmath>

#include "osc.h"

void* callbackfloat(void *originalbuffer, void *newbuffer, int length, void *userdata)
{
  Osc *osc=reinterpret_cast<Osc*>(userdata);
  float *b=static_cast<float*>(newbuffer);
  for(int i=0; i<length && i<osc->bufLen; i++)
    osc->buf[i]=(b[i*2]+b[i*2+1])*.5f;
  return newbuffer;
}

void* callbackshort(void *originalbuffer, void *newbuffer, int length, void *userdata)
{
  Osc *osc=reinterpret_cast<Osc*>(userdata);
  __int16_t *b=static_cast<__int16_t*>(newbuffer);
  for(int i=0; i<length && i<osc->bufLen; i++)
    osc->buf[i]=(b[i*2]+b[i*2+1])/65536.0;
  return newbuffer;
}

Osc::Osc(const string& n) :
  ColorObject(n),
  buf(NULL),
  bufLen(0),
  callback(NULL),
  callbackrunning(false)
{
}

Osc::~Osc()
{
  reset();
  if(callback)
    FSOUND_DSP_Free(callback); 
  if(buf)
    delete buf;
}

void Osc::reset()
{
  if(callbackrunning && callback)
    {
      FSOUND_DSP_SetActive(callback, false);
      callbackrunning=false;
    }
}

int Osc::init()
{
  bufLen=FSOUND_DSP_GetBufferLength();
  if(bufLen<=0)
    {
      cerr << "Osc::init(): FMOD DSP buflen is " << bufLen << endl;
      return -1;
    }
  if(bufLen>800)		// more than 800 pixels of osc don't make much sense
    bufLen=800;
  if(buf)
    delete buf;
  buf=new float[bufLen];
  if(buf==NULL)
    {
      cerr << "Osc::init(): could not alloc buffer " << bufLen << endl;
      return -1;
    }
  for(int i=0; i<bufLen; i++)
    buf[i]=sin(i/100.0);

  switch(FSOUND_GetMixer())
    {
    default:
    case FSOUND_MIXER_BLENDMODE:
    case FSOUND_MIXER_MMXP5:
    case FSOUND_MIXER_MMXP6:
    case FSOUND_MIXER_QUALITY_MMXP5:
    case FSOUND_MIXER_QUALITY_MMXP6:
      callback=FSOUND_DSP_Create(callbackshort, FSOUND_DSP_DEFAULTPRIORITY_CLIPANDCOPYUNIT - 1, this);
      break;
    case FSOUND_MIXER_QUALITY_FPU:
      callback=FSOUND_DSP_Create(callbackfloat, FSOUND_DSP_DEFAULTPRIORITY_CLIPANDCOPYUNIT - 1, this);
      break;
    }

  if(callback==NULL)
    {
      cerr << "Osc::init(): could not init callbackfunc " << FMOD_ErrorString(FSOUND_GetError()) << endl;
      return -1;
    }

  return 0;
}

void Osc::draw()
{
  if(!callbackrunning && callback)
    {
      FSOUND_DSP_SetActive(callback, true);
      callbackrunning=true;
    }

  glBegin(GL_LINE_STRIP);
  glColor4f(r,g,b,a);    
  const float b2=2.0/static_cast<float>(bufLen);
  for(int i=0; i<bufLen; i++)
    glVertex2f((static_cast<float>(i)*b2-1.0), buf[i]);
  glEnd();
}
