#include "background.h"
#include "camera.h"
#include "cube.h"
#include "flash.h"
#include "light.h"
#include "main.h"
#include "scroller.h"
#include "starfield.h"
#include "swoosh.h"

#if MUSIC
#include "osc.h"
#endif

#include "data.h"

#include "doj/gl.hpp"
using namespace doj;

namespace Rotor
{

  static int starttime=-1;
  static Cube *cube=0;
  static Light *light=0;
  static Flash *flash=0;
  static Background *ryg=0;
  static bool flashrun=false;

  static bool cubeenable=false;
  static bool rygenable=false;

#if MUSIC
  static Osc* osc=0;
  static bool oscenable=false;
#endif

  static Camera cam;

  static int sync=-1;

  bool initialized=false;

  static void cleanup()
  {
    if(cube)
      delete cube, cube=0;
    if(light)
      delete light, light=0;
    if(flash)
      delete flash, flash=0;
    if(ryg)
      delete ryg, ryg=0;
#if MUSIC
    if(osc)
      delete osc, osc=0;
#endif
  }

  int init()
  {
    if(initialized)
      return 0;

    atexit(cleanup);

    if(!light)
      {
	light=new Light();
	if(light==NULL)
	  {
	    cerr << "Rotor::init(): could not alloc light" << endl;
	    return -1;
	  }
	light->pos=fvector(1,1,10);
	light->point();
	light->setLight(0);
	if(light->init() < 0)
	  {
	    cerr << "Rotor::init(): could not init light" << endl;
	    return -1;
	  }
      }

    if(!cube)
      {
	cube=new Cube();
	if(cube==NULL)
	  {
	    cerr << "Rotor::init(): could not alloc cube" << endl;
	    return -1;
	  }
	cube->setLight(light);
	cube->colorLight(1, .2, .2, .5, ColorObject::Ambient);
	cube->colorLight(1, .5, .5, .5, ColorObject::Diffuse);
	cube->colorLight(1, .8, .8, .5, ColorObject::Specular);
	cube->shininess=100;
	cube->color(1,1,1,.5);
	cube->blendEnable=true;
	if(cube->init() < 0)
	  {
	    cerr << "Rotor::init(): could not init cube" << endl;
	    return -1;
	  }
      }

    if(!flash)
      {
	flash=new Flash();
	if(flash==NULL)
	  {
	    cerr << "Rotor::init(): could not alloc flash" << endl;
	    return -1;
	  }
	flash->from(0,0,0,0);
	flash->to(0,0,0,1);
	flash->time(1.5);
      }

    if(!ryg)
      {
	ryg=new Background();
	if(ryg==NULL)
	  {
	    cerr << "Rotor::init(): could not alloc ryg" << endl;
	    return -1;
	  }
	ryg->texture(&TGAryg512);
	ryg->blendEnable=true;
	if(ryg->init() < 0)
	  {
	    cerr << "Rotor::init(): could not init ryg" << endl;
	    return -1;
	  }
      }

#if MUSIC
    if(!osc)
      {
	osc=new Osc("Rotor Osc");
	if(osc==NULL)
	  {
	    cerr << "Rotor::init(): could not alloc osc" << endl;
	    return -1;
	  }
	osc->zbufferEnable=false;
	osc->lineWidth=2;
	osc->color(.5, 1, .5, 1);
	if(osc->init() < 0)
	  {
	    cerr << "Rotor::init(): could not init osc" << endl;
	    return -1;
	  }
      }
#endif

    initialized=true;
    return 0;
  }

  void draw(const int time)
  {
    if(!initialized)
      {
	cerr << "Rotor::draw(): not initialized" << endl;
	return;
      }

    for(modSync_t::iterator i=modSync.begin(); i!=modSync.end(); /*nothing here*/)
      {
	const int e=*i>>4;
	if(e==6)
	  {
	    sync=*i&0xf;
	    switch(sync)
	      {
#if MUSIC
	      case 0x1: oscenable=true; break;
#endif
	      case 0x2: cubeenable=true; break;
	      case 0x3: rygenable=true; break;
	      case 0xf: flashrun=true; break;
	      }
	    i=modSync.erase(i);
	  }
	else
	  ++i;
      }

    if(starttime==-1)
      starttime=time;

    const float t=(time-starttime)/1000.0;

    cam.position();

    Starfield::background->textureRotate = Starfield::backgroundSpinSpeed * Starfield::backgroundSpinSpeed * 10.0
      + 10.0 * 2.0 * Starfield::backgroundSpinSpeed * t;
    Starfield::background->drawSingle();

    if(cubeenable)
      {
	light->draw();
	cube->before();
	const float c=::fmod(t, 2.0);
	for(int z=0; z<=20; z++)
	  {
	    const int b=z/3;
	    for(int x=-z-b; x<=z+b; x++)
	      {
		// oben
		glPushMatrix();
		glTranslatef(x, 2, static_cast<float>(3-z) + c);
		glRotatef(t*1.765324, .678, 0.345, 0);
		glScalef(.25,.25,.25);
		cube->draw();
		glPopMatrix();

		// unten
		glPushMatrix();
		glTranslatef(x, -2, static_cast<float>(5-z) - c);
		glRotatef(t*3.765324, .1354, 0, -.2345);
		glScalef(.25,.25,.25);
		cube->draw();
		glPopMatrix();
	      }
	  }
	cube->after();
      }

    if(rygenable)
      {
	static float tt=0;
	if(tt==0)
	  tt=t;
	const float s=fabs(sin(t-tt));
	ryg->color(s,s,s,s);
	ryg->drawSingle();
      }

#if MUSIC
    if(oscenable)
    {
      gluOrthoBefore(screen.width, screen.height);
      {
	glTranslatef(0,0,1);
	osc->drawSingle();
      }
      gluOrthoAfter();
    }
#endif

    if(flashrun)
      {
#if MUSIC
	osc->reset(); // we don't need new data in osc anymore
#endif
	flash->before();
	flashrun=flash->draw(time);
	flash->after();
      }

  }

};
