#include "camera.h"
#include "camerapath.h"
#include "flash.h"
#include "light.h"
#include "main.h"
#include "sphere.h"
#include "sprite.h"
#include "title.h"

#include "data.h"

#include "doj/gl.hpp"
using namespace doj;

#include <sstream>

namespace Space2
{

  static ::Sphere *earth=0;
  static ::Sphere *clouds=0;
  static ::Sphere *stars=0;
  static ::Sphere *sun=0;
  static ::Sphere *mond=0;
  static Sprite *sunflare=0;

  static Light *licht=0;

  static Flash *flashToWhite=0;
  static Flash *flashFromWhite=0;
  static Flash *flashStayWhite=0;
  static Flash *flashToTrans=0;
  static Flash *flashToBlack=0;

  static bool initialized=false;
  static int starttime=-1;

  static Camera cam;
  static CameraPath path;
  static bool pathInit=false;

  static int sync=0;
  static int lastSync=-1;

  static bool effectEnd=false;

  static void deinit()
  {
    if(earth)
      delete earth;
    if(clouds)
      delete clouds;
    if(stars)
      delete stars;
    if(sun)
      delete sun;
    if(sunflare)
      delete sunflare;
    if(mond)
      delete mond;
    if(licht)
      delete licht;
    if(flashToWhite)
      delete flashToWhite;
    if(flashFromWhite)
      delete flashFromWhite;
    if(flashStayWhite)
      delete flashStayWhite;
    if(flashToTrans)
      delete flashToTrans;
    if(flashToBlack)
      delete flashToBlack;
  }

  int init()
  {
    if(initialized)
      return 0;

    atexit(deinit);

    if(!licht)
      {
	licht=new Light("Space2::licht");
	licht->setLight(0);
	licht->init();
      }

    int detail=lowtech?3:5;
    if(getenv("SPHERE"))
      {
	int d=atoi(getenv("SPHERE"));
	if(d>0)
	  detail=d;
      }

    if(!earth)
      {
	earth=new ::Sphere("Space2::earth");
	if(earth==NULL)
	  {
	    cerr << "Space2::init(): could not alloc earth" << endl;
	    return -1;
	  }
	earth->type=::Sphere::Triangle;
	earth->elements=detail;
	earth->texture(&TGAearth2048);
	earth->setLight(licht);
	earth->shininess=1;
	if(earth->init() < 0)
	  {
	    cerr << "Space2::init(): could not init earth" << endl;
	    delete earth;
	    earth=0;
	    return -1;
	  }
      }

    if(!clouds)
      {
	clouds=new ::Sphere("Space2::clouds");
	if(clouds==NULL)
	  {
	    cerr << "Space2::init(): could not alloc clouds" << endl;
	    return -1;
	  }
	clouds->type=::Sphere::Triangle;
	clouds->elements=detail;
	clouds->texture(&TGAclouds2048);
	clouds->blendEnable=true;
	clouds->setLight(licht);
	clouds->shininess=1;
	if(clouds->init() < 0)
	  {
	    cerr << "Space2::init(): could not init clouds" << endl;
	    delete clouds;
	    clouds=0;
	    return -1;
	  }
      }

    if(!stars)
      {
	stars=new ::Sphere("Space2::stars");
	if(stars==NULL)
	  {
	    cerr << "Space2::init(): could not alloc stars" << endl;
	    return -1;
	  }
	stars->radius=70;
	stars->outside=false;
	stars->elements=10;
	stars->texture(&TGAstars256);
	stars->textureModetype=GL_REPLACE;
	stars->textureScale=40;
	if(stars->init() < 0)
	  {
	    cerr << "Space2::init(): could not init stars" << endl;
	    delete stars;
	    stars=0;
	    return -1;
	  }
      }

    if(!sun)
      {
	sun=new ::Sphere("Space2::sun");
	if(sun==NULL)
	  {
	    cerr << "Space2::init(): could not alloc sun" << endl;
	    return -1;
	  }
	sun->radius=.5;
	sun->color(1,1,.7);
	sun->elements=10;
	if(sun->init() < 0)
	  {
	    cerr << "Space2::init(): could not init sun" << endl;
	    delete sun;
	    sun=0;
	    return -1;
	  }
      }

    if(!sunflare)
      {
	sunflare=new Sprite("Space2::sunflare");
	if(sunflare==NULL)
	  {
	    cerr << "Space2::init(): could not init sunflare" << endl;
	    return -1;
	  }
	sunflare->zbufferEnable=false;
	sunflare->blendEnable=true;
	sunflare->texture(&TGAsunflare256);
	if(sunflare->init() < 0)
	  {
	    cerr << "Space2::init(): could not init sunflare" << endl;
	    delete sunflare;
	    sunflare=0;
	    return -1;
	  }
      }

    if(!mond)
      {
	mond=new ::Sphere("Space2::mond");
	if(mond==NULL)
	  {
	    cerr << "Space2::init(): could not alloc mond" << endl;
	    return -1;
	  }
	mond->radius=.25;
	mond->type=::Sphere::Triangle;
	mond->elements=detail-1;
	mond->texture(&TGAmoon256);
	mond->setLight(licht);
	mond->shininess=1;
	if(mond->init() < 0)
	  {
	    cerr << "Space2::init(): could not init mond" << endl;
	    delete mond;
	    mond=0;
	    return -1;
	  }
      }

    if(!flashToWhite)
      {
	flashToWhite=new Flash("Space2::flashToWhite");
	flashToWhite->from(0,0,0,0);
	flashToWhite->to  (1,1,1,0.5);
	flashToWhite->time(1);
      }

    if(!flashFromWhite)
      {
	flashFromWhite=new Flash("Space2::flashFromWhite");
	flashFromWhite->from(1,1,1,0.5);
	flashFromWhite->to  (0,0,0,0);
	flashFromWhite->time(1);
      }

    if(!flashStayWhite)
      {
	flashStayWhite=new Flash("Space2::flashStayWhite");
	flashStayWhite->from(1,1,1,0.5);
	flashStayWhite->to  (1,1,1,0.5);
	flashStayWhite->time(1);
      }

    if(!flashToTrans)
      {
	flashToTrans=new Flash("Space2::flashToTrans");
	flashToTrans->from(0,0,0,1);
	flashToTrans->to  (0,0,0,0);
	flashToTrans->time(5);
      }

    if(!flashToBlack)
      {
	flashToBlack=new Flash("Space2::flashToBlack");
	flashToBlack->from(0,0,0,0);
	flashToBlack->to  (0,0,0,1);
	flashToBlack->time(5);
      }

    {
      std::istringstream s(reinterpret_cast<char*>(&SAVspace2));
      if(s)
	s >> cam;
      else
	{
	  cerr << "Space2::init(): could not init space2.sav stream" << endl;
	  return -1;
	}
    }
    {
      std::istringstream s(reinterpret_cast<char*>(&PTHspace2));
      if(s)
	s >> path;
      else
	{
	  cerr << "Space2::init(): could not init space2.pth stream" << endl;
	  return -1;
	}
      path.init();
    }

    Title::init();

    initialized=true;
    return 0;
  }

  void reset(const int time)
  {
    starttime=time;
    pathInit=false;
    effectEnd=false;
    Title::init();
    flashToWhite->reset();
    flashFromWhite->reset();
    flashStayWhite->reset();
    flashToTrans->reset();
    flashToBlack->reset();
  }

  void draw(const int time)
  {
    if(!initialized)
      init();
    if(starttime==-1)
      reset(time);

    for(modSync_t::iterator i=modSync.begin(); i!=modSync.end(); /*nothing here*/)
      {
	const int e=*i>>4;
	if(e==1)
	  {
	    sync=*i&0xf;
	    i=modSync.erase(i);
	  }
	else
	  ++i;
      }

    if(sync==0 && sync!=lastSync)
      reset(time);

    if(effectEnd)
      return;

    const float t=(time-starttime)/1000.0 - 6.0;

    activeCamera=&cam;
    if(sync>=3)
      {
	if(!pathInit)
	  {
	    path.starttime(time);
	    pathInit=true;
	  }
	path.position(time);
      }
    else
      cam.position();

    clearColor.r=clearColor.g=clearColor.b=clearColor.a=0;

    glPushMatrix();
    {
      glRotatef(t*2.0, 0, 0, 1);
      glTranslatef(0, -30, 0);
      sun->drawSingle();
      licht->drawSingle();
    }
    glPopMatrix();

    glPushMatrix();
    {
      glRotatef(-t, 0, 0, 1);
      earth->drawSingle();
    }
    glPopMatrix();

    glPushMatrix();
    {
      glRotatef(-t*0.8, 0, 0, 1);
      glScalef(1.01, 1.01, 1.01);
      clouds->drawSingle();
    }
    glPopMatrix();

    glPushMatrix();
    {
      glRotatef(-t*.5 -60 , 0, 0, 1);
      glTranslatef(0, 3, 0);
      glRotatef(t*.5, 0, 0, 1);
      mond->drawSingle();
    }
    glPopMatrix();

    glPushMatrix();
    {
      glTranslatef(cam.x, cam.y, cam.z);
      stars->drawSingle();
    }
    glPopMatrix();

    if(t>=2 && t<4.6)
      {
	glPushMatrix();
	{
	  glRotatef(t*2.0, 0, 0, 1);
	  glTranslatef(0, -30, 0);
	  glScale(4);
	  glRotatef(t*15.0, 0, 1, 0);
	  glRotatef(-90, 1, 0, 0);
	  sunflare->drawSingle();
	}
	glPopMatrix();
      }

    if(sync==0)
      {
	flashToTrans->before();
	flashToTrans->draw(time);
	flashToTrans->after();
      }

    if(t>=2 && t<3)
      {
	flashToWhite->before();
	flashToWhite->draw(time);
	flashToWhite->after();
      }

    if(t>=3 && t<4)
      {
	flashStayWhite->before();
	flashStayWhite->draw(time);
	flashStayWhite->after();
      }

    if(t>=4 && t<5)
      {
	flashFromWhite->before();
	flashFromWhite->draw(time);
	flashFromWhite->after();
      }

    if(sync==2)
      Title::draw(time);

    if(sync==4)
      {
	flashToBlack->before();
	effectEnd=!flashToBlack->draw(time);
	flashToBlack->after();
      }

    lastSync=sync;
  }

};
