#include <cstdio>
#include <cstring>
#include <errno.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/types.h>

#ifdef __linux__
#include <sys/mman.h>
#include <unistd.h>
#endif

#include <iostream>
#include <map>
using namespace std;

#include "main.h"
#include <memorymap.hpp>
#include "tga.h"

typedef map<string, GLuint> filecache_t;
static filecache_t filecache;

typedef map<const GLubyte*, GLuint> memcache_t;
static memcache_t memcache;

static int texturesize=0;

int TextureSize()
{
  return texturesize;
}

GLuint makeTexture(const GLubyte *bb,
		   const int width,
		   const int height,
		   const GLenum glformat,
		   const bool mipmap
		   )
{
  // check parameters
  if(bb==0)
    return 0;
  if(width<1)
    return 0;
  if(height<1)
    return 0;

  int bytes=-1;
  switch(glformat)
    {
    case GL_ALPHA:
    case GL_ALPHA4:
    case GL_ALPHA8:
    case GL_LUMINANCE:
    case GL_LUMINANCE4:
    case GL_LUMINANCE8:
    case GL_LUMINANCE4_ALPHA4:
    case GL_LUMINANCE6_ALPHA2:
    case GL_INTENSITY:
    case GL_INTENSITY4:
    case GL_INTENSITY8:
    case GL_RGBA2:
    case GL_R3_G3_B2:
      bytes=1; 
      break;
 
    case GL_ALPHA12:
    case GL_ALPHA16:
    case GL_LUMINANCE12:
    case GL_LUMINANCE16:
    case GL_LUMINANCE_ALPHA:
    case GL_LUMINANCE8_ALPHA8:
    case GL_LUMINANCE12_ALPHA4:
    case GL_INTENSITY12:
    case GL_INTENSITY16:
    case GL_RGB4:
    case GL_RGB5:
    case GL_RGB5_A1:
    case GL_RGBA4:
      bytes=2;
      break;
 
    case GL_LUMINANCE12_ALPHA12:
    case GL_RGB:
    case GL_RGB8:
      bytes=3;
      break;
 
    case GL_LUMINANCE16_ALPHA16:
    case GL_RGB10:
    case GL_RGB12:
    case GL_RGB16:
    case GL_RGBA:
    case GL_RGBA8:
    case GL_RGB10_A2:
    default:
      bytes=4;
      break;
 	      
    case GL_RGBA12:
      bytes=6;
      break;
 
    case GL_RGBA16:
      bytes=8;
      break;
    }

  if(bytes<1)
    return 0;

  // make an opengl texture
  GLuint t=0;
  glGenTextures(1, &t); 
  if(t==0)
    {
      e("makeTexture(): could not generate a new texture");
      return t;
    }

  glBindTexture(GL_TEXTURE_2D, t);

  if(mipmap)
    {
      if(gluBuild2DMipmaps(GL_TEXTURE_2D,
			   glformat,
			   width,
			   height,
			   glformat,
			   GL_UNSIGNED_BYTE, 
			   const_cast<GLubyte*>(bb)) != 0)
	cerr << "could not build mipmaps" << endl;
    }
  else
    {
      glTexImage2D(GL_TEXTURE_2D, 
		   0,	// GLint level
		   bytes,
		   width,
		   height,
		   0,	// GLint border
		   glformat,	// GLenum format
		   GL_UNSIGNED_BYTE,
		   const_cast<GLubyte*>(bb) );
      e("could not build texture");
    }
 
  texturesize+=width*height*bytes;
  return t;
}



GLuint loadTGA(const string& filename, const bool mipmap, const bool cache)
{
#ifdef DOJDEBUG
  clog << "loadTGA(): " << filename << " ";
#endif

  if(cache)
    {
      filecache_t::iterator i=filecache.find(filename);
      if(i!=filecache.end())
	{
#ifdef DOJDEBUG
	  clog << " found in cache" << endl;
#endif
	  return i->second;
	}
    }

  GLubyte *mem=reinterpret_cast<GLubyte*>(memorymap(filename));
  if(mem==0)
    return 0;
  GLuint t=loadTGA(mem, mipmap, false);
  if(t)
    filecache[filename]=t;
  memoryunmap(mem);

  return t;
}


#pragma pack ( push, 1 )
struct head_t
{
  GLubyte idlen;		// 0

  GLubyte cmtype;		// 1

  GLubyte type;			// 2

  GLushort cmstart;		// 3
  GLushort cmlen;		// 5
  GLubyte cmsize;		// 7

  GLushort imgx;		// 8
  GLushort imgy;		// 10
  GLushort width;		// 12
  GLushort height;		// 14
  GLubyte imgbits;		// 15

  GLubyte attrbits :4;		// 16.0
  GLubyte reserved :1;		// 16.4
  GLubyte origin :1 ;		// 16.5
  GLubyte interleave :2 ;	// 16.6

  GLubyte data;
};
#pragma pack ( pop )

GLuint loadTGA(const GLubyte *memory, const bool mipmap, const bool cache)
{
  if(memory==0)
    return 0;

  if(cache)
    {
      memcache_t::iterator i=memcache.find(memory);
      if(i!=memcache.end())
	{
#ifdef DOJDEBUG
	  clog << "found in cache" << endl;
#endif
	  return i->second;
	}
    }

  // check header
  const head_t *head=reinterpret_cast<const head_t*>(memory);
  const int type=static_cast<int>(head->type);
  if(type==2 || type==10)
    {
#ifdef DOJDEBUG
      clog << (char*)((type==10)?" RLE":"");
#endif

      int width=static_cast<int>(head->width);
      int height=static_cast<int>(head->height);
      GLenum TGAglformat=0;

      if(static_cast<int>(head->imgbits)==24 && static_cast<int>(head->attrbits)==0)
	{
	  TGAglformat=GL_RGB;
#ifdef DOJDEBUG
	  clog << " RGB";
#endif
	}
      if(static_cast<int>(head->imgbits)==32 && static_cast<int>(head->attrbits)==8)
	{
	  TGAglformat=GL_RGBA; 
#ifdef DOJDEBUG
	  clog <<" RGBA";
#endif
	}

      if(!TGAglformat || width<1 || height<1)
	{
	  cerr << "loadTGA(): can't load:";
	  cerr << " imgbits: " << static_cast<int>(head->imgbits);
	  cerr << " attrbits: " << static_cast<int>(head->attrbits);
	  cerr << " width: " << width;
	  cerr << " height: " << height;
	  cerr << endl;
	  return 0;
	}

#ifdef DOJDEBUG
      const int origin=static_cast<int>(head->origin);
      clog << " origin: " << origin;
#endif

      // bytes per pixel
      const int bytes=head->imgbits/8;

      // sucking NVIDIA cards fail with BGR modes. So we have to convert tga manually
      GLubyte *data=new GLubyte[width*height*bytes];
      if(data==0)
	{
	  cerr << "no memory for image data" << endl;
	  return 0;
	}

      const GLubyte *img=&(head->data);
      if(type==2)
	{
	  // convert colors to network byte order
	  if(bytes==4)
	    for(int i=0; i<width*height*bytes; i+=bytes)
	      {
		data[i+0]=img[i+2];
		data[i+1]=img[i+1];
		data[i+2]=img[i+0];
		data[i+3]=img[i+3];
	      }
	  else
	    for(int i=0; i<width*height*bytes; i+=bytes)
	      {
		data[i+0]=img[i+2];
		data[i+1]=img[i+1];
		data[i+2]=img[i+0];
	      }
	}
      else if(type==10) // rle compressed
	{
	  int i=0; 
	  int o=0;
	  while(o<width*height*bytes)
	    {
	      const GLubyte header=img[i++];
	      int count=(header&0x7f) + 1;
	      if(header&0x80)
		{
		  const GLubyte b=img[i++];
		  const GLubyte g=img[i++];
		  const GLubyte r=img[i++];
		  const GLubyte a=(bytes==4)?img[i++]:0;
		  while(count--)
		    {
		      data[o++]=r;
		      data[o++]=g;
		      data[o++]=b;
		      if(bytes==4)
			data[o++]=a;
		    }
		}
	      else
		{
		  while(count--)
		    {
		      data[o++]=img[i+2];
		      data[o++]=img[i+1];
		      data[o++]=img[i+0];
		      i+=3;
		      if(bytes==4)
			data[o++]=img[i++];
		    }
		}
	    }
	}

      int nw=0, nh=0;
      switch(TextureMemorySize)
	{
	case 8: nw=nh=256; break;
	case 16: nw=nh=512; break;
	default: break;
	}

      // check if we need to scale
      if(nw && nh)
	{
	  GLubyte *d=new GLubyte[nw*nh*bytes];
	  if(data)
	    {
	      if(gluScaleImage(TGAglformat, width, height, GL_UNSIGNED_BYTE, data, nw, nh, GL_UNSIGNED_BYTE, d) != 0)
		e("loadTGA(): error while scaling!");

	      delete [] data;
	      data=d;
	      width=nw;
	      height=nh;
	    }
	  else
	    cerr << "no memory for image data" << endl;
	}

      GLuint t=makeTexture(data, width, height, TGAglformat, mipmap);
      delete [] data;
      if(t)
	memcache[memory]=t;

      return t;
    }
  else
    {
      cerr << " wrong image type: " << static_cast<int>(head->type);
    }
  return 0;
}
